#include "ti86asm.inc"

.org _asm_exec_ram

	nop
	jp start
	.dw 0
	.dw description

getkey:
	call _getky
	or a
	jr z,getkey
	ret

displines:
	ld de,98*256+1
	call _ILine
	ld de,1*256+62
	jp _ILine

horiz_line:
	ld e,c
	call _ILine
	ld a,c
	add a,l
	ld c,a
	ret

MultHL:
	xor a
	cp h
	jr z,SetZero
	cp l
	jr z,SetZero
	push bc
	call _HtimesL
	pop bc
	ret
SetZero:
	ld hl,0
	ret

; modified routine originally by Matthew Shepcar
GetA:
	ld l,a
	ld h,0
GetHL:
	xor a
	ld de,-1
	ld (_curRow),de
	push bc
	call $4a33
	pop bc
	dec hl
DiscardSpaces:
	ld a,(hl)
	inc hl
	cp ' '
	jr z,DiscardSpaces
	dec hl
	ret

DispA:
	ld l,a
	ld h,0
	jr DispHL

DispAStatus:
	ld l,a
	ld h,0
DispHLStatus:
	ld c,$72
DispHL:
	call GetHL
	push bc
	push hl
	ld bc,10
	xor a
	cpir			;bc=9-size
	ld a,9
	sub c
	pop hl
	pop bc
	add a,a		;multiply by 2
	cpl
	add a,c		;center column
	ld c,a		;load into current column

vputs:
	ld (_penCol),bc
	jp _vputs

split:						;copy from $fc00 to $f700
	di
	ld b,128				;number of pixels across
split_loop:
	push bc
	ld de,$fc1f-32
	ld hl,$f700
	ld a,128
	sub b
	and %11111000
	rra \ rra \ rra			;divide by 8 to get # of bytes to shift de
	ld c,a
	ld b,0
	push bc
	ex de,hl
	sbc hl,bc
	ex de,hl
	exx
	ld de,$fc00-32
	ld hl,$f70f-48
	pop bc
	ex de,hl
	add hl,bc
	ex de,hl
	ld c,32				;number of rows / 2
split_screen:
	push bc
	ld bc,32
	ex de,hl
	add hl,bc
	ex de,hl
	ld c,48
	add hl,bc
	pop bc
	ld b,16
	ld a,(de)
	rlca
	ld (de),a
split_left:
	rl (hl)
	dec hl
	djnz split_left
	exx
	ld bc,32
	ex de,hl
	add hl,bc
	ex de,hl
	ld c,16
	add hl,bc
	ld b,c				;c=16 already
	ld a,(de)
	rrca
	ld (de),a
split_right:
	rr (hl)
	inc hl
	djnz split_right
	exx
	dec c
	jr nz,split_screen
	exx
	pop bc
	djnz split_loop
	ret

copy_screen:			;just copy to video mem
	ld hl,$fc00
	ld de,$f700
	ld bc,1024
	ldir
	ret

draw_spaces:		;display d spaces at (b,c)
	push af
	push bc
	push hl
	ld (_penCol),bc
space_loop:
	push de
	call _vputspace
	pop de
	dec d
	jr nz,space_loop
	pop hl
	pop bc
	pop af
	ret

draw_level:
;display the level
;shamelessly ripped from 86 central
	ld hl,current_level
	ld bc,$0202	;start drawing at (2, 2)
level_loop:
	push hl
	ld l,(hl)		;get the current tile
	ld h,0
	add hl,hl		;* 2
	ld d,h
	ld e,l
	add hl,hl		;* 4
	add hl,hl		;* 8
	add hl,de		;* 10
	add hl,de		;* 12
	add hl,de		;* 14
	ld de,bricks	;load the start of the sprites
	add hl,de		;add together to get correct sprite
	call PutSprite	;draw the sprite
	pop hl		;restore map pointer
	inc hl			;go to next tile
	ld a,b
	add a,12
	ld b,a
	cp 98		;we're done
	jr nz,level_loop	;only jump if the row isn't complete
	ld b,2		;draw at the left again
	ld a,c
	add a,6
	ld c,a		;move location down a row
	cp 62		;check if it's done
	jr nz,level_loop	;jump back to the top if not done
	ret

check_level:
;nz if not a diamonds level
	rst 10h				;do a _findsym to get bde
	ret c
	ld a,b
	ex de,hl
	call _ahl_plus_2_pg3
	call _getb_ahl			;get first byte and hl->string now
	cp $dd
	ret nz
	inc hl	
	ld a,(hl)				;get second byte
	dec a				;cp $01
	ret

start:
	ld (save_sp),sp			;save stack pointer
	ld (level_num),a			;a=0, so first level
	ld hl,_plotSScreen
	ld (hl),a				;clear first byte
	ld de,_plotSScreen+1
	ld bc,1023
	ldir
	call copy_screen
	ld a,$37
	out (0),a				;move screen
	call _runindicoff
	call _flushallmenus

title_screen:
	ld sp,0
save_sp = $-2

	xor a
	ld (first_screen),a
	ld hl,titlepic
	ld de,$fc00
	call DispRLE
	ld hl,name
	ld bc,$1028
	call vputs


	ld hl,level_name			;load previously saved level
	rst 20h
	rst 10h
	jr c,get_new_level		;level doesn't exist
	call check_level
	jr z,finish_title_screen		;valid level, so we just keep it
get_new_level:
	ld hl,starting_name
	rst 20h				;copy starting varname to op1
find_first_level:
	rst 08h				;op1 to op2
	xor a
	call _FindAlphaUp		;search for string
	jr c,no_levels			;no strings
	call check_level
	jr z,finish_level_detect		;valid level
;now we check if we're getting anywhere in this search
	ld hl,_OP1+1
	ld de,_OP2+1
	call _strcmp			;see if name has changed
	jr nz,find_first_level		;keep searching if name is changing

no_levels:
	ld hl,error_txt
	ld bc,$2518
	call vputs
	call split
	call getkey
	jp exit

finish_level_detect:
	ld hl,_OP1
	ld de,level_name
	call _mov10b			;save level name
finish_title_screen:
	ld hl,menu_txt
	ld bc,$1a2c
	call vputs
	ld bc,$222c
	call vputs
	ld bc,$2a2c
	call vputs
	ld bc,$322c
	call vputs
	ld bc,$3a2c
	call vputs
	ld bc,0
	ld (score),bc			;starting score
update_level:
	ld bc,$323a
	push bc				;save coordinates
	ld d,70
	call draw_spaces		;overwrite previous description
	ld hl,level_name
	rst 20h				;move to op1
	rst 10h				;_findsym
	ld a,b
	ex de,hl
	call _get_word_ahl		;de=size, ahl+=2
	push de
	call _ahl_plus_2_pg3
	call _set_abs_src_addr
	pop hl
	xor a
	call _set_mm_num_bytes	;size to copy
	ld hl,highscore
	call _set_abs_dest_addr	;copy here for now
	call _mm_ldir
	ld hl,num_levels
	pop bc				;retrieve description
	call vputs				;display description
	ld a,(hl)				;get number of levels
	push hl
	ld h,a
	ld l,42
	call MultHL			;get size
	inc hl
	push hl
	ld de,num_levels
	pop bc				;retrieve size
	pop hl				;retrieve pointer to end of string
	ldir					;copy over description
	ld bc,$3a52
	push bc
	ld d,40
	call draw_spaces
	pop bc
	ld hl,initials
	call vputs				;display initials
	call _vputspace
	call _vputspace			;for spacing
	ld hl,(highscore)
	call GetHL
	call _vputs				;display after initials
update_speed:
	ld a,(speed)
	ld bc,$2a55
	call DispA				;display speed
update_level_num:
	ld bc,$224c
	push bc
	ld d,15
	call draw_spaces		;overwrite last level number
	ld a,(level_num)
	inc a
	call GetA
	pop bc				;retrieve coordinates
	call vputs
	ld hl,first_screen
	ld a,(hl)
	ld (hl),1
	or a
	call z,split
	call copy_screen
get_selections:
	call getkey
	cp K_EXIT
	jp z,exit
	cp K_F1
	jp z,start_level
	cp K_F2
	jr z,speed_up
	cp K_F3
	jr z,get_next_level
	dec a				;cp K_DOWN
	jr z,level_down
	cp K_UP-1
	jr nz,get_selections

level_up:
	ld hl,level_num
	inc (hl)
	ld a,(num_levels)
	cp (hl)				;check if we're at the max
	jr nz,update_level_num
	ld (hl),0				;first level
	jr update_level_num

level_down:
	ld hl,level_num
	dec (hl)
	ld a,(hl)
	inc a					;see if we're at -1
	jr nz,update_level_num
	ld a,(num_levels)
	ld (hl),a				;go to max
	jr level_down			;now decrement it once

speed_up:
	ld hl,speed
	inc (hl)
	ld a,(hl)
	cp 6
	jp nz,update_speed
	ld (hl),1
	jr update_speed

get_next_level:
	xor a
	ld (level_num),a			;reset level number for new level
	ld hl,level_name
	rst 20h				;mov10toop1
detect_level:
	ld de,temp_search_name
	call _movfrop1
	xor a
	call _FindAlphaUp		;search for string
	ld hl,_OP1+1
	ld de,temp_search_name+1
	call _strcmp
	jr z,get_first_level		;same var, so we're at end of alphabet.
	call check_level
	jr nz,detect_level		;invalid level
save_new_level:
	ld de,level_name
	call _movfrop1			;copy new name
	jp update_level
;same name, so we load default name and find first level
get_first_level:
	ld hl,starting_name
	rst 20h				;copy starting varname to op1
find_a_level:
	xor a
	call _FindAlphaUp
	call check_level
	jr nz,find_a_level			;keep searching until we find something
	jr save_new_level

start_level:
	ld a,5
	ld (lives),a				;starting lives
	ld hl,speed_table
	ld a,(speed)
	add a,a
	ld e,a
	ld d,0
	add hl,de				;add offset
	call $33				;get regular and bonus delays
	ld (delay_amount),hl
	call _clrLCD
	ld h,1
	ld bc,$0101	
	call displines
	ld bc,98*256+62
	call displines
	ld bc,99*256+1
	ld d,126
	ld l,11
	call horiz_line
	ld l,8
	call horiz_line
	ld l,14
	call horiz_line
	ld l,14
	call horiz_line
	ld l,14
	call horiz_line
	call horiz_line
	ld bc,126*256+1
	call _ILine
	ld hl,bonus_msg
	ld bc,$0267
	call vputs				;"Bonus"
	ld b,$10
	call vputs				;"Level"
	ld b,$1e
	call vputs				;"Score"
	call draw_bonus
	xor a
	ld b,$24
	call DispAStatus		;display starting score
	call draw_lives

new_level:
	ld a,(level_num)
	ld h,a
	ld l,42
	call MultHL
	ld de,levels
	add hl,de				;hl->level data
	ld de,start_coords
	ldi
	ldi					;copy ball coordinates
	ld b,40				;40 bytes to copy
	xor a
decomp_level:
	rld
	ld (de),a
	inc de
	rld
	ld (de),a
	inc de
	inc hl
	djnz decomp_level
	ld a,61
	ld (bonus),a			;the bonus score for the round
	ld bc,256+1			;set bonus delay, direction=down
	ld (bonus_delay),bc
	xor a
	ld (hit_key),a			;not active yet
	ld (reverse_mode),a		;regular keys
	ld bc,(start_coords)
	ld (ball_y),bc
	call draw_level			;display the level
	ld a,(level_num)
	inc a
	ld b,$16
	call DispAStatus		;display the level
	ld a,1
	ld (brick_mode),a		;white bricks
	call draw_brick_mode		;display brick mode
	call draw_bonus			;display bonus
	call draw_ball			;display the ball
	call split				;copy to screen

new_life:
	call copy_screen		;show ball
	call start_delay			;delay with ball showing

;************************************************main loop

main_loop:
	call check_bricks		;check how many bricks left
	call anim_ball			;draw the ball
	call draw_bonus			;display the bonus
	call check_keys			;check for keys
	call delay				;speed control
	call copy_screen		;copy buffer to display
	jr main_loop

;************************************************update lives
draw_lives:
	ld a,(lives)
	ld d,a
	ld bc,$642d
ball_draw_loop:
	ld hl,ball
	call PutSprite
	ld a,b
	add a,5
	ld b,a
	dec d
	jr nz,ball_draw_loop
	ret

;************************************************update bonus
draw_bonus:
	ld hl,bonus_delay
	dec (hl)
	ret nz
	ld a,(bonus)
	or a
	ret z
	ld a,(bonus_delay_amount)
	ld (hl),a				;new counter
bonus_skip_delay:			;skip delay
	ld bc,$086d
	ld hl,bonus
	dec (hl)
	ld a,(hl)
	cp 9
	jr nz,skip_spaces
	ld d,7
	call draw_spaces		;overwrite old bonus with spaces
skip_spaces:
	jp DispAStatus			;display the bonus

;************************************************update score
draw_score:
;increment score and display it
	ld hl,(score)
	inc hl
	ld (score),hl
	push hl
	ld b,$24
	call DispHLStatus
	pop hl
	ld a,(lives)
	cp 5
	ret z					;can't have more than 5 lives
	ld de,400
sub_loop:
	or a
	sbc hl,de
	ret c					;not divisible by 400
	jr nz,sub_loop			;not done yet
;divisible by 400, so we get an extra life
	call draw_lives
	ld hl,lives
	inc (hl)
	jr draw_lives			;draw new lives

;************************************************check keys
check_keys:
	di					;no down-left bug (not that they need to press that anyway)
	ld a,%01111110
	out (1),a
	in a,(1)
	rra \ 	rra				;left
	push af
	call nc,move_left
	pop af
	rra					;right
	call nc,move_right
	ld a,%01111101
	out (1),a
	in a,(1)
	bit 6,a				;clear
	jp z,lose_life
	cpl
	and 24
	cp 24
	jr z,invert
	ld a,%00111111
	out (1),a
	in a,(1)
	rla					;more
	jr nc,pause
	rla
	jp nc,check_hiscore		;exit
	rla
	ret c					;2nd

shutdown:
	ld a,1
	out (3),a
	ei
	halt
	ld a,11
	out (3),a

start_delay:
	xor a
start_delay_outer:
	ld b,0
start_delay_inner = $-1		;use the 0 as a nop :-)
	nop \ nop \ nop
	djnz start_delay_inner
	dec a
	jr nz,start_delay_outer
	ret

pause:
	call getkey
	cp K_PLUS
	jr z,contrast_up
	cp K_MINUS
	ret nz

contrast_down:
	ld hl,_contrast
	ld a,(hl)
	or a
	jr z,pause
	dec (hl)
	jr set_contrast

contrast_up:
	ld hl,_contrast
	ld a,(hl)
	cp 31
	jr z,pause
	inc (hl)

set_contrast:
	ld a,(hl)
	out (2),a
	jr pause

invert:
	ld hl,$fc00
	ld bc,4				;4*256=1024
invert_loop:
	ld a,(hl)
	cpl
	ld (hl),a
	inc hl
	djnz invert_loop
	dec c
	jr nz,invert_loop
	ld hl,$c3ea				;text flags
	ld a,(hl)
	xor %00001000			;toggle inverted text
	ld (hl),a
	ret

;************************************************check high score
check_hiscore:
	res 3,(iy+5)			;not inverted text
	ld hl,(score)
	ld de,(highscore)
	call _cphlde
	jp c,title_screen			;no high score
	jp z,title_screen
	ld (highscore),hl			;save new high score
	call _clrScrn
	ld hl,high_score
	ld bc,$0d1d
	call vputs
	ld bc,$162b
	call vputs
	call split
	ld bc,$0904
	ld (_curRow),bc			;set coordinates for input
	ld hl,initials
	ld b,0

;modified string input routine from ztetris
;originally by Jimmy Mardell
get_string:
	push hl
	push bc
	call copy_screen
	call getkey
	pop bc
	cp K_DEL
	jr z,backspace
	cp K_ENTER
	jr z,name_done
	cp K_SIGN
	jr nz,checkletter
	ld a,' '
	pop hl
	jr putletter
checkletter:
	ld hl,letters
	push bc
	ld bc,26
	cpir				;search for letter
	ld d,c
	pop bc
	pop hl
	jr nz,get_string
	ld a,65
	add a,d
putletter:
	ld c,a
	ld a,b
	cp 3				;3 letters for initials
	jr z,get_string
	ld (hl),c
	inc hl
	inc b
	ld a,c
	call _putc
	jr get_string
backspace:
	pop hl
	ld a,b
	or a
	jr z,get_string
	dec b
	dec hl
	push hl
	ld a,' '
	ld (hl),a
	ld hl,_curCol
	dec (hl)
	call _putc
	dec (hl)
	pop hl
	jr get_string
name_done:
	pop hl
	ld (hl),0				;null terminator
;now we save it back to the level
	ld hl,level_name
	rst 20h
	rst 10h
	xor a
	ld hl,4
	add hl,de
	adc a,b
	ld b,6				;6 bytes
	ld de,highscore
	call save_data
	jp title_screen

;************************************************exit
exit:
	ld hl,varname-1
	rst 20h				;copy to OP1
	rst 10h				;_findsym
	xor a
	ld hl,data_start-_asm_exec_ram+4		;offset
	add hl,de				;hl=pointer to data in original prog
	adc a,b				;in case we overlapped pages
	ld de,data_start			;copy from here
	ld b,data_end-data_start
	call save_data
	ld a,$3c
	out (0),a				;restore video mem
	res 4,(iy+9)
	jp _clrWindow

;************************************************save data
save_data:
	;ahl->dest
	;de->src
	;b=num bytes
	push af
	push hl
	call _getb_ahl
	ld a,(de)
	ld (hl),a
	pop hl
	pop af
	call _inc_ptr_ahl
	inc de
	djnz save_data
	ret

;************************************************move ball
draw_ball:
	ld hl,ball
	ld bc,(ball_y)
	jp PutSprite

anim_ball:
	xor a
	ld (collision),a		;default is no change
	ld a,(ball_y)
	cp 1
	jr z,bounce
	cp 58
	jr nz,no_bounce
bounce:
	ld a,(direction)
	neg
	ld (direction),a
no_bounce:
	call draw_ball		;erase old ball
	ld a,(direction)		;get ball direction
	add a,c			;add to y-coordinate
	ld (ball_y),a		;update coordinate
	call draw_ball		;redraw ball at new coordinates
	push bc
	dec b			;left side
	call check_col
	pop bc
	inc b				;right side
	call check_col
	ld a,(collision)
	or a
	ret z				;no collision
	ld a,(direction)
	neg
	ld (direction),a		;new direction
	ret

move_horiz:
	xor a
	ld (collision),a		;no collision yet
	ld bc,(ball_y)
	ld a,b
	ret

move_left:
	ld a,(reverse_mode)
	or a
	jr nz,move_right_nocheck
move_left_nocheck:
	call move_horiz
	dec a
	cp 1
	ret z
	ld b,a
	dec b			;left side
	call check_col
	ret nz			;a collision, so we don't move
	call draw_ball		;erase old ball
	ld hl,ball_x
	dec (hl)
	jp draw_ball		;draw new ball

move_right:
	ld a,(reverse_mode)
	or a
	jr nz,move_left_nocheck
move_right_nocheck:
	call move_horiz
	inc a
	cp 94
	ret z
	ld b,a
	inc b				;right side
	call check_col
	ret nz			;a collision, so we don't move
	call draw_ball		;erase old ball
	ld hl,ball_x
	inc (hl)
	jp draw_ball		;draw new ball

check_col:
	;nz if collision
	push bc
	dec c			;top
	call check_collision
	pop bc
	inc c				;bottom row
	call check_collision
	ld a,(collision)
	or a
	ret

check_collision:
	;checks collision at (b,c)
	;clears brick if necessary
	;activates collision if a collision

	call GetTile
	ret c				;a clear tile
	ld (collision),a		;collision (a greater than 0 anyway)
	ld a,(brick_mode)
	cp (hl)
	jr z,clear_tile		;remove the tile if same mode
	ld a,(hl)
	sub 6
	ret c				;a regular brick which we can't break
	jr nz,not_key
	inc a
	ld (hit_key),a		;1=key active
	jr clear_tile			;remove the key
not_key:
	dec a
	jr nz,not_lock
	ld a,(hit_key)
	or a
	ret z
	jr clear_tile			;remove lock
not_lock:
	dec a
	jp z,lose_life		;a killer brick
	dec a
	ret z				;solid brick (never breaks)
	cp 4
	jr nz,not_reverse
	ld de,reverse_mode
	ld a,(de)
	xor 1				;toggle reverse mode
	ld (de),a
	jr clear_tile
not_reverse:
	push af
	ld a,(brick_mode)
	cp 5
	jr nz,normal_brick
	pop af
	ret
normal_brick:
	call draw_brick_mode	;remove old mode
	pop af
	inc a
	ld (brick_mode),a	;new brick mode
	jp draw_brick_mode	;draw new mode


clear_tile:
	;erase tile pointed to by hl
	ld a,(hl)
	push af
	ld (hl),0
	ld de,current_level
	or a
	sbc hl,de
	ld a,8
	call _divHLbyA		;separate x and y dimensions
;l=y, a=x
	ld b,a
	ld h,6
	call MultHL
	ld c,l				;y-coordinate
	ld l,b
	ld h,12
	call MultHL
	ld b,l				;x-coordinate
	inc b
	inc b
	inc c
	inc c
	pop hl
	ld l,14
	call MultHL
	ld de,bricks
	add hl,de
	call PutSprite		;clear the tile from the screen
	jp draw_score		;increment score and quit

; bc = coords to get tile at (x, y)
; returns: hl = pointer to map tile
; tile = (((y-2) / 6) * 8) + ((x-2) / 12)
GetTile:
	ld h,0
	ld l,b
	ld a,12
	push bc
	call _divHLbyA
	pop bc
	ld b,l			;store x value
	ld l,c
	ld a,6		;divide by 6
	push bc
	call _divHLbyA
	pop bc
	ld a,l			;a=y coordinate
	add a,a
	add a,a
	add a,a		;y*8
	add a,b		;add to x
	ld hl,current_level
	ld e,a
	ld d,0
	add hl,de		;get offset of tile
	ld a,(hl)
	cp 1			;c flag if solid
	ret

draw_brick_mode:
	ld a,(brick_mode)
	ld l,a
	ld h,14
	call MultHL
	ld de,bricks
	add hl,de
	ld bc,$6a36
	jp PutSprite			;draw the brick

;************************************************check bricks
check_bricks:
	ld de,0				;d=number of regular bricks, e=number of weird bricks
	ld hl,current_level
	ld b,80				;size of level
bricks_loop:
	ld a,(hl)
	inc hl
	or a
	jr z,bricks_next_loop		;empty brick
	cp 8
	jr nc,bricks_next_loop
	inc d
	cp 5					;check if it's an ending brick
	jr nz,bricks_next_loop	
	dec d
	inc e
bricks_next_loop:
	djnz bricks_loop

	ld a,d
	or e
	jr z,done_level

	ld a,d
	or a
	ret nz				;still some regular bricks

	ld a,(brick_mode)
	cp 5
	ret z

	call draw_brick_mode
	ld a,5
	ld (brick_mode),a
	jp draw_brick_mode

done_level:
	ld a,(bonus)
	or a
	jr z,done_bonus
	ld a,14
	call start_delay_outer		;use another delay routine
	call bonus_skip_delay	;decrement bonus
	call draw_score			;increment score
	call copy_screen
	jr done_level
done_bonus:
	ld hl,level_num
	ld a,(hl)
	ld b,a
	ld a,(num_levels)
	dec a
	cp b
	jp z,check_hiscore		;played all levels
	inc (hl)
	call draw_level			;erase level
	call draw_ball			;erase ball
	call draw_brick_mode		;erase brick mode
	jp new_level

;************************************************delay
delay:
	ld a,(delay_amount)
delay_loop_outer:
	ld b,30
delay_loop_inner:
	nop
	djnz delay_loop_inner
	dec a
	jr nz,delay_loop_outer
	ret

;************************************************dead
lose_life:
	call draw_ball			;erase ball
	call draw_lives			;remove lives
	ld hl,lives
	dec (hl)
	jp z,check_hiscore		;dead
	call draw_lives
	ld bc,(start_coords)
	ld (ball_y),bc
	ld a,1
	ld (direction),a
	call draw_ball
	call draw_brick_mode
	ld a,1
	ld (brick_mode),a
	call draw_brick_mode
	jp new_life

; =============================================================
; Puts a sprite stored at (HL) at B,C
; by Matt Johnson <matt@acz.org>
; =============================================================
PutSprite:

	push bc				; Save BC (X, Y) 
	push de				; Save DE
	push hl				; Save HL (Start of Sprite Data)
	push hl				; Save HL (Start of Sprite Data)

	push bc
	ld d, c
	ld e, b
	call FindPixel				; Finds pixel at E, D
	pop bc
				; PutSprite16 needs pixel at B, C
  
				; A = Bitmask with one bit set; this will be the bit (pixel) "counter"

	ex de,hl				; DE = Address in Display of Sprite
	pop hl				; HL = Start of Sprite Data
	ld b,(hl)				; B = X width of Sprite 
	inc hl					; Next byte
	ld c,(hl)				; C = Y width of Sprite
	inc hl					; Next byte (now pointing at actual sprite data)
	push hl				; Save the twice incremented HL into stack
	pop ix				; IX = The recently twice incremented HL

	ex de,hl				; HL = Address in Display of Sprite

PS_NewRow:
	push bc				; Save X width of Sprite and Y width of Sprite
	ld d,(ix)				; D = (IX), so D = First byte from Sprite
	inc ix
	ld e,(ix)				; E = (IX), so E = Second byte from Sprite
	inc ix					; IX points to next row

	push af				; Save Bitmask 
	push hl				; Save HL, Address in Display of Sprite

PS_NewCol:				; Now the fun begins, remember A is the bitmask, D = the Left Column Byte from Sprite, E = the Right Column Byte from Sprite
	sla e					; 16-bit rotation DE
	rl d

	push af				; Save Bitmask
	jr nc,PS_NewPixel		; Bit not set

	xor (hl)				; Invert Pixel at the "current bit" (Bitmask is the "bit counter")
	ld (hl),a				; (HL) = A, Save changes to

PS_NewPixel:
	pop af				; Restore Bitmask
	rrca					; A is rotated right *through* carry
	jr nc,PS_SameByte		; If the carry was set, that means that one bit set in A (bit counter) 
					;  rotated all the way to the end right into carry and recycles back into Bit 7
					;  so it can be used for the next byte

	inc hl					; Move on to next byte

PS_SameByte:
	djnz PS_NewCol		; B = X (width of sprite), so it loops to PS_NewCol X times. This means that is X = 6, 
				; it will Shift Right the bitmask (move the bit counter to the right) 6 times, comparing 
				; each bit of the bitmask and the sprite and setting or clearing the pixel in that
				; particular bit. It then moves on the the next pixel.

				; Move on to the next row
	pop hl				; Recover HL, the Address in Display of the Sprite
	pop af				; Recover AF, A = Bitmask
	ld de,16				; DE = 16
	add hl,de				; HL = HL + 16, this moves down ONE row in the Display Area (128 width / (8 bits/pixel) = 16 bytes)
	pop bc				; Recover X_width and Y_height of Sprite
	dec c				; C = Y_height of Sprite, subract one, which means one row of the Sprite has been drawn

	jr nz,PS_NewRow		; If there are more rows to be drawn, go back to PS_NewRow

; No more rows. Since there were "effectively" 3 pushes before PS_NewRow, there must be three pops that way the
; ret statement will retrieve the correct address when it returns to the calling program.

	pop hl                                         
	pop de
	pop bc
	ret

; =============================================================
; Dan Eble & James Yopp FindPixel routine
; Input:  D = y, E = x
; Output: HL= addr in vid mem, A = bitmask, C is modified
; =============================================================
FindPixel:
        ld hl,FP_Bits
        ld a,e
        and $07         ; a = bit offset
        add a,l
        ld l,a
        adc a,h
        sub l
        ld h,a
        ld c,(hl)       ; c = bitmask for (hl)
;48 t-states up to this point
        ld hl,FP_RLD
        ld (hl),d
        ld a,e          ; a = x/8 (byte offset within row)
        rrca
        rrca   
        rrca
        rld
        or $FC 
        ld l,(hl)
        ld h,a          ; hl -> byte in vid mem
        ld a,c          ; now a = bitmask for (hl)
;121 t-states up to this point
        ret

FP_RLD:  .db $00
FP_Bits: .db $80,$40,$20,$10,$08,$04,$02,$01

;===========================================================
; RLE picture displayer v1.1
; Decodes a RLE picture made by RLE2PIC
;
; written by David Phillips <electrum@tfs.net>
; started: 8/19/98
; last update: 11/5/98
;
; input: HL = RLE encoded picture, DE = where to display
; output: 1024 byte decoded picture
; destroys: AF, BC, DE, HL
; current size: 32 bytes
;===========================================================
DispRLE:
	ld bc,1024			; we need to copy 
DispRLEL:
	ld a,(hl)			; get the next byte
	cp $91			; is it a run?
	jr z,DispRLERun	; then we need to decode the run
	ldi				; copy the byte, and update counters
DispRLEC:
	ld a,b			; check the low byte and
	or c				; the high byte for 0
	jr nz,DispRLEL		; if not, then we're not done either
	ret				; if it's zero, we're done
DispRLERun:
	inc hl				; move to the run value
	ld a,(hl)			; get the run value
	inc hl				; move to the run count
	push hl			; save source pointer
	ld h,(hl)			; get the run count
	ex de,hl			; swap source and destination pointers
DispRLERunL:
	ld (hl),a			; copy the byte
	inc hl				; increase destination pointer
	dec bc			; decrease byte count
	dec d			; decrease run count
	jr nz,DispRLERunL	; if we're not done, then loop
	ex de,hl			; swap pointers back
	pop hl			; recover source pointer
	inc hl				; advance the source pointer
	jr DispRLEC		; check to see if we should loop


ball:
	.db 5, 5
	.db %01110000,0
	.db %11111000,0
	.db %11111000,0
	.db %11111000,0
	.db %01110000

bricks:		;start of bricks
;0
	.db 12, 6
	.db %00000000, %00000000
	.db %00000000, %00000000
	.db %00000000, %00000000
	.db %00000000, %00000000
	.db %00000000, %00000000
	.db %00000000, %00000000

;1
	.db 12, 6
	.db %01111111, %11100000
	.db %10000000, %00010000
	.db %10000000, %00010000
	.db %10000000, %00010000
	.db %10000000, %00010000
	.db %01111111, %11100000

;2
	.db 12, 6
	.db %01111111, %11100000
	.db %10101010, %10110000
	.db %11010101, %01010000
	.db %10101010, %10110000
	.db %11010101, %01010000
	.db %01111111, %11100000

;3
	.db 12, 6
	.db %01111111, %11100000
	.db %10000000, %00010000
	.db %10111111, %11010000
	.db %10111111, %11010000
	.db %10000000, %00010000
	.db %01111111, %11100000

;4
	.db 12, 6
	.db %01111111, %11100000
	.db %10100100, %10010000
	.db %10010010, %01010000
	.db %11001001, %00110000
	.db %10100100, %10010000
	.db %01111111, %11100000

;5
	.db 12, 6
	.db %01111111, %11100000
	.db %11111111, %11110000
	.db %11100000, %01110000
	.db %11100000, %01110000
	.db %11111111, %11110000
	.db %01111111, %11100000

;6
	.db 12, 6
	.db %01111111, %11100000
	.db %11111110, %00110000
	.db %10000000, %10110000
	.db %11001110, %00110000
	.db %11111111, %11110000
	.db %01111111, %11100000

;7
	.db 12, 6
	.db %01111111, %11100000
	.db %11111001, %11110000
	.db %11110110, %11110000
	.db %11110000, %11110000
	.db %11110000, %11110000
	.db %01111111, %11100000

;8
	.db 12, 6
	.db %01111111, %11100000
	.db %11100111, %11110000
	.db %10000000, %00010000
	.db %10000000, %00010000
	.db %11100111, %11110000
	.db %01111111, %11100000

;9
	.db 12, 6
	.db %01111111, %11100000
	.db %11011110, %11110000
	.db %11111011, %11110000
	.db %11101111, %10110000
	.db %11111101, %11110000
	.db %01111111, %11100000

;10
	.db 12, 6
	.db %01111111, %11100000
	.db %10101011, %11110000
	.db %11010101, %11110000
	.db %10101011, %11110000
	.db %11010101, %11110000
	.db %01111111, %11100000

;11
	.db 12, 6
	.db %01111111, %11100000
	.db %10000001, %11110000
	.db %10111111, %11110000
	.db %10111111, %11110000
	.db %10000001, %11110000
	.db %01111111, %11100000

;12
	.db 12, 6
	.db %01111111, %11100000
	.db %10100101, %11110000
	.db %10010011, %11110000
	.db %11001001, %11110000
	.db %10100101, %11110000
	.db %01111111, %11100000

;13
	.db 12, 6
	.db %01111111, %11100000
	.db %11111111, %10110000
	.db %11011100, %00010000
	.db %10000011, %10110000
	.db %11011111, %11110000
	.db %01111111, %11100000



varname:
	.db 8
description:
	.db "Diamonds 1.2 "
name:
	.db "by Jonah Cohen",0
menu_txt:
	.db "F1 - Start",0
	.db 30,31," Level:",0			;30=up, 31=down
	.db "F2 - Speed:",0
	.db "F3 -",0
	.db "High score:",0

error_txt:
	.db "Error: no levels detected",0

bonus_msg:
	.db "Bonus",0
level_msg:
	.db "Level",0
score_msg:
	.db "Score",0

high_score:
	.db "You have a high score!",0
	.db "Enter initials:",0

speed_table = $-2
	.db 200,20,140,30,75,50,45,70,30,90

letters:		;used in string input
	.db $19,$21,$0a,$12,$1a,$22,$0b,$13,$1b,$23,$2b
	.db $0c,$14,$1c,$24,$2c,$0d,$15,$1d,$25,$2d,$0e
	.db $16,$1e,$26,$2e


starting_name:
	.db $0c,1,0

data_start:

level_name:
	.db $0c,7,"dlevels",0,0		;store the current level name

speed:
	.db 3

data_end:


titlepic:
 .db $91,$00,$04,$f0,$c0,$91,$00,$05,$c0,$91,$00,$08
 .db $f8,$c0,$91,$00,$05,$c0,$91,$00,$08,$dc,$91,$00
 .db $06,$c0,$91,$00,$08,$ce,$c3,$de,$f0,$1e,$1e,$03
 .db $c3,$c0,$91,$00,$07,$c6,$c7,$df,$f8,$3f,$1f,$07
 .db $c7,$c0,$91,$00,$07,$c6,$ce,$db,$dc,$73,$9b,$8e
 .db $ce,$91,$00,$08,$c6,$dc,$d9,$ce,$e1,$d9,$dc,$dc
 .db $91,$00,$08,$c6,$d8,$d8,$c6,$c0,$91,$d8,$03,$91
 .db $00,$08,$c6,$d8,$d8,$c6,$c0,$d8,$d8,$df,$c0,$91
 .db $00,$07,$c6,$d8,$d8,$c6,$c0,$d8,$d8,$df,$c0,$91
 .db $00,$07,$c6,$d8,$d8,$c6,$c0,$d8,$d8,$c0,$c0,$91
 .db $00,$07,$ce,$dc,$d8,$c6,$e1,$d8,$dc,$c1,$c0,$91
 .db $00,$07,$dc,$ce,$d8,$c6,$73,$98,$ce,$c3,$80,$91
 .db $00,$07,$f8,$c7,$d8,$c6,$3f,$18,$c7,$df,$91,$00
 .db $08,$f0,$c3,$d8,$c6,$1e,$18,$c3,$de,$91,$00,$ff
 .db $91,$00,$ff,$91,$00,$ff,$91,$00,$17

first_screen		equ		$		;0=first time at title screen
level_num			equ		$+1		;which level we're on
score			equ		$+2		;player's score (2 bytes)
ball_y			equ		$+4		;y-coordinate
ball_x			equ		$+5		;x-coordinate
delay_amount		equ		$+6		;regular delay speed
bonus_delay_amount	equ		$+7		;bonus delay speed
bonus			equ		$+8		;the bonus score
bonus_delay		equ		$+9		;countdown for bonus
direction			equ		$+10		;1=down, -1=up
reverse_mode		equ		$+11		;1=reversed directions
brick_mode		equ		$+12		;code of current brick type
lives				equ		$+13		;number of balls left
collision			equ		$+14		;1=collision
hit_key			equ		$+15		;whether they hit the key yet
temp_search_name	equ		$+16		;name of var in search (11 bytes)
start_coords		equ		$+27		;initial coordinates for level (2 bytes)
current_level		equ		$+29		;decompressed level (80 bytes)
highscore			equ		$+109	;the high score (2 bytes)
initials			equ		$+111	;initials for high score (4 bytes)
num_levels		equ		$+115	;number of levels
levels			equ		$+116	;the entire level string (unlimited size)

.end
