#include "ti86asm.inc"

;Colony 1.0 by Cassady Roop
;SImmoralTech Software
;You are a budding bacterium colony trying to get a foothold...  Wierd, eh?
;A cell dies of overcrowding with =>[max_n] neighbors in any direction,
;dies of isolation with <[min_n] neighbors, and is created in an empty cell with =[opt_n] neighbors.
;Played on a 10x10 grid.
;Coords are represented as 0-9 [y,x].
;X increases to the right from zero at left. Y increases to the bottom from 0 at the top.
;Array offsets are any number 0-100 pointing to coords [0,0]-[9,9].

max_n				equ			4					;this or more neighbors than this, cell dies
min_n				equ			2					;fewer neighbors than this, cell dies
opt_n				equ			3					;this many neighbors, cell created
max_levels			equ			20					;highest level number available
max_moves			equ			100					;highest number of moves allowed before a forced losegame

dynamic_data		equ			$EAD0
cells				equ			dynamic_data+0		;number of occupied grid units, 0-100.  1 byte
moves				equ			dynamic_data+1		;number of moves that have been made. 1 byte
score				equ			dynamic_data+2		;score total.  Cheating zeroes this. 2 bytes
curX				equ			dynamic_data+4		;cursor X coord. 1 byte
curY				equ			dynamic_data+5		;cursor Y coord. 1 byte
curXold				equ			dynamic_data+6		;old curX
curYold				equ			dynamic_data+7		;old curY
level				equ			dynamic_data+8		;current level. 1 byte
cells_needed		equ			dynamic_data+9		;cells needed before level advancement. 1 byte
cur_grid			equ			dynamic_data+10		;array describing current grid.  100 bytes
new_grid			equ			cur_grid+100		;array describing new grid.  100 bytes


.org _asm_exec_ram

	nop
	jp ProgStart
	.dw 0
	.dw titletext

ProgStart:

	call _clrLcd
	call _runindicoff
	call _flushallmenus
	xor a
	ld (cells), a
	ld (moves), a
	ld (curX), a
	ld (curY), a
	ld (curXold), a
	ld (curYold), a
	ld (level), a
	ld (cells_needed), a
	ld (score), a
	ld (score+1), a
	call get_highscore			;gets high score from prog var in case prog was double-loaded
	call dispMain				;displays mainscreen text
	call load_nextLevel			;loads the next level
	call draw_grid				;creates graphical grid on screen from grid mem space.
	call dispCursor				;display cursor
	call Update_Level			;update entire statistics display
	jp keyloop





dispMain:							;display mainscreen text
	ld hl, $0058
	ld (_pencol), hl
	ld hl, titletext
	call _vputs						;display title
	ld hl, $076A
	ld (_pencol), hl
	ld hl, secondtext
	call _vputs						;'F1'
	ld hl, $116A
	ld (_pencol), hl
	ld hl, f1text
	call _vputs						;'Help'

	ld de, $FC8B
	ld hl, arrowSprite				;pic of four arrows
	ld bc, 16						
	ld a, 15						;sprite is 23 lines high, 2 bytes wide
dispMain_spriteloop:
	push de
	push bc
	ld bc, 2
	ldir							;move 3 sprite bytes to screen
	pop bc
	pop de
	ex de, hl
	add hl, bc						;next line
	ex de, hl
	dec a
	cp 0
	jp nz, dispMain_spriteloop

	ld hl, $1E52
	ld (_pencol), hl
	ld hl, leveltext
	call _vputs						;'Level'
	ld a, ':'
	call _vputmap
	ld hl, $2452
	ld (_pencol), hl
	ld hl, goaltext
	call _vputs						;'Goal:'
dispStat:
	ld hl, $3052
	ld (_pencol), hl
	ld hl, movestext
	call _vputs						;'Moves:'
	ld hl, $3652
	ld (_pencol), hl
	ld hl, cellstext
	call _vputs						;'Cells: '
	ret


clr_grid:							;clear the grid
	ld hl, cur_grid			
	ld b, 100
clr_gridLoop:
	ld (hl), 0
	inc hl
	djnz clr_gridloop
	ret



draw_grid:
	ld de, cur_grid					;start of current grid array
	ld bc, $0000					;grid coord (0,0)
d_g_draw:
	ld a, (de)						;get byte from grid array for this grid position
	inc de							;point to next array position
	cp 0							;zero denotes empty (openSprite)
	jp z, d_g_drawOpen
	ld hl, cellSprite				;sprite of an occupied grid space
	jp d_g_drawSprite				;jp used instead of jr 'cause it is faster
d_g_drawOpen:
	ld hl, openSprite				;sprite of a vacant grid space
d_g_drawSprite:
	call simSprite					;draw the sprite
	ld a, c							;get x coord
	cp 9							;is it at max?
	jp z, d_g_incY					;if so, we'll set it to zero, and increment the Y coord
	inc c							;otherwise, increment the X coord
	jp d_g_draw						;draw the next sprite
d_g_incY:
	xor a
	ld c, a							;zero the X coord
	ld a, b							;get Y coord
	cp 9							;is it at max?
	jp z, d_g_drawDone				;if so, we drew the last sprite (9,9)
	inc b							;otherwise, increment Y coord
	jp d_g_draw						;draw the next sprite
d_g_drawDone:						;sprites that make up the grid are all drawn
	ld hl, $FC0A					;upper right corner of grid
	ld b, 61						;drawing a vertical line 61 pixels long to make right-hand edge line
	ld de, 16
d_g_rVert:							;draw vertical line at right
	ld a, (hl)						;get screen byte
	or %10000000					;turn on leftmost pixel in that byte
	ld (hl), a						;put it back
	add hl, de
	djnz d_g_rVert					;loop [b] times
	ld hl, $FFC0
	ld b, 10
d_g_lHoriz:							;draw horizontal line at bottom		
	ld (hl), $FF
	inc hl
	djnz d_g_lHoriz					;loop [b] times
	ret




keyloop:
	call _get_key
	cp 0
	jr z, keyloop
	cp k_right
	jp z, right					;->  = move right
	cp k_left
	jp z, left					;<-  = move left
	cp k_up
	jp z, up					;UP = move up
	cp k_down
	jp z, down					;DOWN = move down
	cp k_second
	jp z, click					;2ND = select a grid square
	cp k_exit
	jp z, check_hiscore			;see if player has a high score, otherwise exit
	cp k_f1
	jp z, help					;F1 = Help
	cp k_more
	jp z, incLevel				;MORE = next level
	jr keyloop
incLevel:						;this is a 'cheat' that increments the level number
	ld hl, $0000
	ld (score), hl				;cheating gets you a zero for a score
	call clr_grid				;clear the board
	call load_nextLevel			;load the next level
	call draw_grid				;display it
	call dispCursor				;show the cursor
	call Update_level			;update all stats
	jp keyloop




right:
	ld a, (curX)				;get current X coord
	cp 9						;at max?
	jp z, keyloop				;if so, can't do it
	ld (curXold), a				;the current will be the old
	ld a, (curY)
	ld (curYold), a
	call undispCursor			;undisplays the old cursor and replaces background
	ld a, (curX)
	inc a						;increment X coord
	ld (curX), a				;update X coord
	call dispCursor				;display the cursor
	jp keyloop


left:
	ld a, (curX)				;get current X coord
	cp 0						;at min?
	jp z, keyloop				;if so, can't do it
	ld (curXold), a				;the current becomes the old
	ld a, (curY)
	ld (curYold), a
	call undispCursor			;undisplay this cursor
	ld a, (curX)
	dec a
	ld (curX), a				;update coord
	call dispCursor				;display new cursor
	jp keyloop


up:
	ld a, (curY)				;get current Y coord
	cp 0						;at min?
	jp z, keyloop				;if so, no can do
	ld (curYold), a				;the current becomes the old
	ld a, (curX)
	ld (curXold), a
	call undispCursor			;undisplay current cursor
	ld a, (curY)
	dec a
	ld (curY), a				;update
	call dispCursor				;show the change
	jp keyloop


down:
	ld a, (curY)				;get current Y coord
	cp 9						;at max?
	jp z, keyloop				;if so, no can do
	ld (curYold), a				;the current becomes the old
	ld a, (curX)
	ld (curXold), a
	call undispCursor			;undisplay current cursor
	ld a, (curY)
	inc a
	ld (curY), a				;update
	call dispCursor				;show the change
	jp keyloop

	

click:
	ld a, (curX)
	ld c, a
	ld a, (curY)
	ld b, a
	ld hl, cur_grid
	push bc						;save
	call get_cell				;find out what is already there
	xor 1						;the opposite
	pop bc
	call set_cell				;put the opposite into the grid at current coords
	call eval_grid				;evaluates grid and creates next iteration in new_grid
	ld hl, new_grid
	ld de, cur_grid
	ld bc, 100
	ldir						;copy new iteration to current grid
	call draw_grid				;show the new grid
	ld a, (moves)
	inc a
	ld (moves), a				;increment move counter
	cp max_moves				;max amount of moves?
	jp z, loseGame				;then you've made too many moves
	call count_cells			;get number of cells
	ld (cells), a				;save that
	cp 0						;zero cells?
	jp z, loseGame				;then you lose!
	ld b, a
	ld a, (cells_needed)		;number of cells that is the goal
	cp b						;have we reached the goal?
	jp z, winGame				;if exactly the goal, win game
	sub b
	jp m, winGame				;if over the goal, win game
	call Update_stats			;update statistics display
	call dispCursor
	jp keyloop


loseGame:
	call _clrLCD				;clear the screen
	call draw_grid				;show the grid
	ld hl, $0C5E
	ld (_pencol), hl
	ld hl, colonytext
	call _vputs					;'Colony'
	ld hl, $125C
	ld (_pencol), hl
	ld hl, hasdiedtext
	call _vputs					;'Has Died'
	call dispStat				;displays 'Cells:' and 'Moves:' headings
	call Update_stats			;updates cell and moves count displays
	ld a, max_moves
	ld (moves), a				;to give a score of 0 for this level
	call dispPoints				;displays 'Points: xxx'
l_g_keyloop:
	call _get_key
	cp 0
	jr z, l_g_keyloop
	jp check_hiscore			;save score if it is a high score, else quit


winGame:
	call _clrLCD				;clear the screen
	call draw_grid				;show the grid
	ld hl, $0C5E
	ld (_pencol), hl
	ld hl, leveltext
	call _vputs					;'Level'
	ld hl, $1258
	ld (_pencol), hl
	ld hl, completetext
	call _vputs					;'Complete'
	call dispStat				;displays 'Cells:' and 'Moves:' headings
	call Update_stats			;updates cell and moves count displays
	call dispPoints				;displays 'Points: xxx', updates score, and disps score
w_g_keyloop:
	call _get_key
	cp 0
	jr z, w_g_keyloop
	cp k_exit
	jp z, _clrLCD
	call _clrLCD
	call dispMain				;display text stuff
	call load_nextLevel			;load the next level's data
	call draw_grid				;display it
	call dispCursor				;show the cursor
	call Update_level			;update all statistical counters
	jp keyloop


check_hiscore:					;see if old hiscore was beaten.  If so get initials, else exit
	ld hl, (score)				;get score for this game
	ld de, (highscore)			;get highest score so far
	xor a						;res carry
	ld a, d						;MSB of old score
	ld b, h						;MSB of new score
	sub b						;subtract new from old
	jr z, check_hiscore_lsb		;if MSB's are equal, check lower byte
	jr c, newRecord				;if carry, then the old is less than the new
	jp _clrLCD					;otherwise, quit
check_hiscore_lsb:				;check lower byte
	ld a, e						;lsb of old
	ld b, l						;lsb of new
	sub b						;sub new from old
	jr c, newRecord				;if carry, then old is less than the new
	jp _clrLCD					;getting the same score, or lower, doesn't quite cut it
newRecord:						;new high score, so record it to the original program var
	call _clrLCD
	ld hl, $0C00
	ld (_pencol), hl
	ld hl, haveatext
	call _vputs					;'You have a '
	ld hl, hightext
	call _vputs					;'High '
	ld hl, scoretext
	call _vputs					;'Score'
	ld a, '!'
	call _vputmap				;'!'
	ld hl, $1200
	ld (_pencol), hl
	ld hl, scoretext
	call _vputs					;'Score'
	ld a, ':'
	call _vputmap
	ld a, ' '
	call _vputmap
	ld hl, (score)
	call displayHL	 			;display the score
	ld hl, $1E00
	ld (_pencol), hl
	ld hl, enterinitialstext
	call _vputs					;'Enter initials:'
	ld a, 5
	call _vputmap				;disp an arrow symbol
	ld hl, highscore+2			;we'll store the string here
	ld a, ' '
	ld (hl), a
	inc hl
	ld (hl), a
	inc hl
	ld (hl), a					;three space chars
	inc hl
	ld hl, highscore+2			;destination
	ld b, 3						;max chars
	set shiftAlpha, (IY+shiftflags)	;set Alpha
	set shiftAlock, (IY+shiftflags)	;set AlphaLock
	call textIn					;text input, modified version of the routine
	res shiftAlpha, (IY+shiftflags)	;unset alpha lock
	jr nz, newRecord				;if canceled with LEFT key, redisplay all (left=backspace)
	ld hl, progname-1
	rst 20h
	rst 10h
	ret c						;bad things'll happen if the var was renamed and it doesn't exist by this name anymore
	ld a, b
	ex de, hl
	call decodeABS
	add a, 64
	add hl, de
	out (5), a
	inc a
	out (6), a
	inc hl
	inc hl						;skip length bytes
	inc hl
	inc hl						;skip ASMprgm token
	ld de, highscore-_asm_exec_ram		;offset into var to get to the highscore data
	add hl, de					;point to highscore
	ex de, hl
	ld hl, (score)
	ld a, l						;LSB
	ld (de), a
	inc de
	ld a, h						;MSB
	ld (de), a					;score is now copied
	inc de
	ld hl, highscore+2			;we stored initials here
	ld bc, 3					;copy 3 letters
	ldir						;copy the initials to the prog var
	ld a, $0D
	out (5), a					;put call table back
	jp _clrLCD



dispPoints:
	ld hl, $1D52
	ld (_pencol), hl
	ld hl, pointstext
	call _vputs					;'Points: '
	call find_points
	call AtoDEC
	call Update_score			;compute new score
	ld hl, $2452
	ld (_pencol), hl
	ld hl, scoretext
	call _vputs					;'Score'
	ld a, ':'
	call _vputmap
	ld a, ' '
	call _vputmap
	ld hl, (score)
	call displayHL
	ret


get_highscore:					;gets high score info from prog var
	ld hl, progname-1			;program name
	rst 20h						;move to OP1
	rst 10h						;_findsym
	ret c						;if not by original name, too bad, exit
	ld a, b
	ex de, hl					;ld ahl, bde
	call decodeABS				;get asic address
	add hl, de					;put hl into RAM page window
	add a, 64					;set bit 6 for RAM page
	out (5), a					;load
	inc a
	out (6), a					;load
	inc hl
	inc hl						;skip length bytes
	inc hl
	inc hl						;skip asmprgm token
	ld de, highscore-_asm_exec_ram	;offset into prog
	add hl, de
	ld de, highscore			;destination
	ld bc, 5					;data length
	ldir						;copy
	ld a, $0d
	out (5), a					;put ROM call table back
	ret



;-----------------------------------------
eval_grid:						;evaluates grid and creates new iteration in new_grid
	ld hl, cur_grid				;the grid being evaluated
	ld de, new_grid				;grid being written to
	ld a, 0						;offset counter
eval_cell:						;evaluates a cell and moves on to next cell
	cp 100
	ret z						;done if it is 100
	push af						;save offset counter
	call get_coords				;creates coords in [bc] from offset in [a]. [a],[hl],[de] preserved.
	ld d, 0						;d is now the neighbor counter
eval1:							;check cell above/left of center
	dec b						;move up
	dec c						;move left
	call chValid				;see if those are valid coords. Z if invalid.
	jp z, eval2					;if invalid, eval next neighbor cell
	call get_cell					;content of that cell output to [a]. [hl],[bc] preserved.
	cp 0						;is it an empty neighbor?
	jp z, eval2					;if so, check the next one
	inc d						;neighbor counter inc'd, because we found a neighbor
eval2:
	inc b						;move down
	call chValid				;see if those are valid coords. Z if invalid.
	jp z, eval3					;if invalid, eval next neighbor cell
	call get_cell					;content of that cell output to [a]. [hl],[bc] preserved.
	cp 0						;is it an empty neighbor?
	jp z, eval3					;if so, check the next one
	inc d						;neighbor counter inc'd, because we found a neighbor
eval3:
	inc b						;move down
	call chValid				;see if those are valid coords. Z if invalid.
	jp z, eval4					;if invalid, eval next neighbor cell
	call get_cell					;content of that cell output to [a]. [hl],[bc] preserved.
	cp 0						;is it an empty neighbor?
	jp z, eval4					;if so, check the next one
	inc d						;neighbor counter inc'd, because we found a neighbor
eval4:
	inc c						;move right
	call chValid				;see if those are valid coords. Z if invalid.
	jp z, eval5					;if invalid, eval next neighbor cell
	call get_cell					;content of that cell output to [a]. [hl],[bc] preserved.
	cp 0						;is it an empty neighbor?
	jp z, eval5					;if so, check the next one
	inc d						;neighbor counter inc'd, because we found a neighbor
eval5:
	dec b						;move up
	dec b						;move up
	call chValid				;see if those are valid coords. Z if invalid.
	jp z, eval6					;if invalid, eval next neighbor cell
	call get_cell					;content of that cell output to [a]. [hl],[bc] preserved.
	cp 0						;is it an empty neighbor?
	jp z, eval6					;if so, check the next one
	inc d						;neighbor counter inc'd, because we found a neighbor
eval6:
	inc c						;move right
	call chValid				;see if those are valid coords. Z if invalid.
	jp z, eval7					;if invalid, eval next neighbor cell
	call get_cell					;content of that cell output to [a]. [hl],[bc] preserved.
	cp 0						;is it an empty neighbor?
	jp z, eval7					;if so, check the next one
	inc d						;neighbor counter inc'd, because we found a neighbor
eval7:
	inc b						;move down
	call chValid				;see if those are valid coords. Z if invalid.
	jp z, eval8					;if invalid, eval next neighbor cell
	call get_cell					;content of that cell output to [a]. [hl],[bc] preserved.
	cp 0						;is it an empty neighbor?
	jp z, eval8					;if so, check the next one
	inc d						;neighbor counter inc'd, because we found a neighbor
eval8:
	inc b						;move up
	call chValid				;see if those are valid coords. Z if invalid.
	jp z, eval_neigh_done		;if invalid, eval next neighbor cell
	call get_cell					;content of that cell output to [a]. [hl],[bc] preserved.
	cp 0						;is it an empty neighbor?
	jp z, eval_neigh_done		;if so, check the next one
	inc d						;neighbor counter inc'd, because we found a neighbor
eval_neigh_done:				;all neighboring cells checked. [d]=neighbors found.
	dec b
	dec c						;return coords to original, centered position
	call get_cell				;find out if the cell is occupied or not
	cp 0						;unoccupied?
	jp z, eval_Opt_check		;if so, we don't need to worry about anything but optimal creation condition
	ld a, d						;get number of neighbors
	sub min_n					;subtract the minimum number of neighbors required for that cell's survival
	jp m, eval_kill				;if neg, then fewer than minimum neighbors, so cell dies
	ld a, d
	sub max_n					;subtract maximum number of neighbors for a cell to survive
	jp p, eval_kill				;if pos, then too many neighbors, so cell dies
eval_survive:
	push hl
	ld hl, new_grid
	ld a, 1
	call set_cell				;set cell at [bc] in grid [hl] to value [a]
	pop hl
	pop af
	inc a						;increment the cycle counter
	jp eval_cell				;reloop
eval_kill:
	push hl
	ld hl, new_grid
	xor a
	call set_cell
	pop hl
	pop af
	inc a
	jp eval_cell
eval_Opt_check:					;see if a cell should be created
	push hl
	ld hl, new_grid
	ld a, d						;number of neighbors
	cp opt_n					;is it the optimal number of neighbors required for cell creation?
	jp z, eval_Opt_create		;yes, it is
	xor a						;zero byte
	call set_cell				;sets cell at [bc] in grid at [hl] to value [a]
	pop hl
	pop af						;the cycle counter
	inc a						;increment it
	jp eval_cell				;do the whole thing over again, 100 times...	
eval_Opt_create:
	ld a, 1
	call set_cell				;sets cell at [bc] in grid at [hl] to value [a]
	pop hl
	pop af						;the cycle counter
	inc a						;increment it
	jp eval_cell		

chValid:						;check coords in [bc] for validity. Z if invalid.
	ld a, b						;get Y coord
	cp -1
	ret z
	cp 10
	ret z
	ld a, c
	cp -1
	ret z
	cp 10
	ret							;either ret with Z (invalid) or NZ (valid

get_coords:						;outputs coords [bc][y,x] for offset in [a]
	push af						;preserve offset
	ld b, 0						;Y coord counter
get_c_loop:
	sub 10						;subtract one Y
	jp m, get_c_ones			;if neg, then we have no more tens
	inc b						;Y coord
	jr get_c_loop				;reloop
get_c_ones:
	add a, 10					;add back ten to get pos number less than ten
	ld c, a						;get the X coord into c
	pop af						;we preserved af
	ret							;[y,x] is now in [bc]

get_offset:						;outputs offset [a] for coords in [bc][y,x]. [bc] preserved.
	push bc						;preserve coords
	xor a						;set to zero and res carry flag
	ld a, b						;get Y coord
	rla							;*2
	ld b, a						;hold that
	rla							;*4
	rla							;*8
	add a, b					;*8+*2=*10
	add a, c					;add X coord to get offset
	pop bc						;we preserved coords
	ret							;offset is now in [a]
;-----------------------------------------



help:
	ld hl, $FC00
	ld de, _plotSscreen
	ld bc, 1024
	ldir							;copy screen
	set graphDraw, (IY+graphFlags)	;force a redraw next time graph screen is used
	call _clrLCD
	ld hl, $0001
	ld (_pencol), hl
	ld hl, titletext
	call _vputs						;disp title
	ld hl, $004B
	ld (_pencol), hl
	ld hl, credittext
	call _vputs						;'by Cassady Roop'
	ld hl, $381A
	ld (_pencol), hl
	ld hl, credit2text
	call _vputs						;'SImmoralTech Software'
	ld hl, $0601
	ld (_pencol), hl
	ld hl, hightext
	call _vputs						;'High '
	ld hl, scoretext
	call _vputs						;'Score'
	ld a, ':'
	call _vputmap					;':'
	ld a, ' '
	call _vputmap					;' '
	ld hl, (highscore)				;get highscore word
	call displayHL					;display the score
	ld a, ' '
	call _vputmap
	ld a, ' '
	call _vputmap
	ld hl, highscore+2				;start of initials string
	call _vputs						;display initials to go with score
	ld hl, $FC00
	ld b, $D0
help_dark1:
	ld a, (hl)
	cpl
	ld (hl), a
	inc hl
	djnz help_dark1
	ld hl, $FF80
	ld b, $70
help_dark2:
	ld a, (hl)
	cpl
	ld (hl), a
	inc hl
	djnz help_dark2
	ld hl, $1200
	ld (_pencol), hl
	ld hl, helpthetext
	call _vputs						;'Help the '
	ld hl, colonytext
	call _vputs						;'Colony'
	ld hl, growtext
	call _vputs						;' Grow!'
	ld hl, $1800
	ld (_pencol), hl
	ld hl, secondtext
	call _vputs						;'[2ND]'
	ld hl, $1815
	ld (_pencol), hl
	ld hl, addremovetext
	call _vputs						;'Add/Remove a Cell'
	ld hl, $2300
	ld (_pencol), hl
	ld hl, cellsdietext
	call _vputs						;'Cells die with '
	ld a, 25						;char code for greater-than-or-equal sign
	call _vputmap
	ld a, max_n
	call AtoDEC						;disp max neighbors
	ld hl, neighborstext
	call _vputs						;'neighbors'
	ld hl, $2900
	ld (_pencol), hl
	ld hl, cellsdietext
	call _vputs						;'Cells die with '
	ld a, 60						;char code for less-than sign
	call _vputmap
	ld a, min_n
	call AtoDEC						;disp min neighbors
	ld hl, neighborstext
	call _vputs						;'neighbors'
	ld hl, $2F00
	ld (_pencol), hl
	ld hl, newcellstext
	call _vputs						;'New cells appear with '
	ld a, opt_n
	call AtoDEC						;disp optimum conditions number
	ld hl, neighborstext
	call _vputs						;'neighbors'

help_keyloop:
	call _get_key
	cp 0
	jr z, help_keyloop
	ld hl, _plotSscreen
	ld de, $FC00
	ld bc, 1024
	ldir							;copy screen back
	jp keyloop




dispCursor:							;displays cursor
	ld a, (curX)					;get current x posit
	ld c, a
	ld a, (curY)					;get current y posit
	ld b, a							;[bc]=[y,x]
	push bc							;save that
	ld hl, cur_grid
	call get_cell					;find out what's in that grid cell
	cp 0
	jr nz, dispCursor_cell
	ld hl, cursorOpenSprite			;cursor to use if the square is open
	jr dispCursor_sprite
dispCursor_cell:
	ld hl, cursorCellSprite			;cursor to use if the square is occupied
dispCursor_sprite:
	pop bc							;retrieve coords
	call simSprite					;display the cursor
	ret


undispCursor:						;displays what was there before the cursor (when it has moved away)
	ld a, (curXold)
	ld c, a
	ld a, (curYold)
	ld b, a
	push bc							;save that
	ld hl, cur_grid
	call get_cell					;find out what's in that grid cell
	cp 0
	jr nz, undispCursor_cell
	ld hl, OpenSprite				;if the square is open
	jr undispCursor_sprite
undispCursor_cell:
	ld hl, CellSprite				;if the square is occupied
undispCursor_sprite:
	pop bc							;retrieve coords
	call simSprite					;display the sprite
	ret


set_cell:							;places a byte [a] at [bc][y,x] in grid mem pointed to by hl.  preserves hl.
	push hl							;this routine preserves hl
	push af
	ld a, b							;get Y coord
	and a							;res the carry flag
	rla								;*2
	ld b, a
	rla								;*4
	rla								;*8
	add a, b						;*8+*2=*10
	add a, c						;add X coord to get offset into grid array
	ld b, 0
	ld c, a
	add hl, bc						;get address of that array entry
	pop af
	ld (hl), a						;places input byte [a] into array entry pointed to by coords [bc]
	pop hl							;this routine preserves hl
	ret
	
	
get_cell:							;given [bc][y,x] and grid at [hl] (preserved), returns value of cell in [a]
	push hl
	push bc							;preserve coords
	ld a, b							;get Y coord
	and a							;res the carry flag
	rla								;*2
	ld b, a
	rla								;*4
	rla								;*8
	add a, b						;*8+*2=*10
	add a, c						;add X coord to get offset into grid array
	ld b, 0
	ld c, a
	add hl, bc						;get address of that array entry
	ld a, (hl)
	pop bc
	pop hl
	ret


count_cells:						;returns number of active cells found in cur_grid in [a]
	ld hl, cur_grid
	ld b, 100
	ld c, 0
count_cells_loop:
	ld a, (hl)
	cp 0
	jr nz, count_cells_loop2
	inc hl
	djnz count_cells_loop
	jr count_cells_done
count_cells_loop2:
	inc c
	inc hl
	djnz count_cells_loop
count_cells_done:
	ld a, c
	ret


find_points:						;finds points for that level, returned in [a]
	ld a, (moves)					;number of moves made in this level
	ld b, a
	ld a, max_moves					;max possible moves
	sub b							;compute points.  Many moves=low score, few moves=high score.
	ret


update_score:						;computes score for that level, returned in [a], and update [score].
	call find_points				;returns points for this level in [a]
	ld hl, (score)					;get running total of points
	add a, l						;add to low byte
	ld l, a							;put it back
	jr nc, update_score_updated
	inc h							;if there was a carry over into the high byte
update_score_updated:					;hl now contains the updated score total
	ld (score), hl					;store it
	ret


load_nextLevel:						;loads the next level
	ld a, (level)
	cp max_levels
	jr z, l_nl_wrap					;don't exceed highest level number
	inc a
	ld (level), a
	jp load_level
l_nl_wrap:
	ld a, 1
	ld (level), a					;set to first level
load_level:							;loads level number in [level] to cur_grid. (Up to 127 levels possible)
	call clr_grid					;first clear the cur_grid and new_grid
	ld a, (level)					;get level number
	and a							;res carry
	dec a							;using 0+ offsets, so level 1 is found at index+0
	rla								;multiply by 2
	ld c, a
	ld b, 0
	ld hl, LevelIndex
	add hl, bc						;point to index pointer for desired level
	ld e, (hl)
	inc hl
	ld d, (hl)
	ex de, hl						;hl points to start of level data
	ld a, (hl)						;number of cells needed to pass this level
	ld (cells_needed), a			;save that
	inc hl
	ld a, (hl)						;number of cells present in level
	ld (cells), a					;save that
	ld b, a							;counter
load_levelLoop:
	inc hl	
	ld a, (hl)
	push hl
	ld hl, cur_grid
	ld d, 0
	ld e, a							;offset
	add hl, de
	ld (hl), 1						;set cell to 1
	pop hl
	djnz load_levelLoop				;do that [b] times
	xor a
	ld (moves), a					;new level, so no moves yet
	ret


Update_Level:								;updates the number displays concerning current level, and stats
	ld hl, $1E6B
	ld (_pencol), hl
	push hl
	call UpdateClr
	pop hl
	ld (_pencol), hl
	ld a, (level)
	call AtoDEC						;display level number
	ld hl, $246B
	ld (_pencol), hl
	push hl
	call UpdateClr
	pop hl
	ld (_pencol), hl
	ld a, (cells_needed)
	call AtoDEC						;display number of cells needed to advance level
Update_stats:						;only updates running stat numbers
	ld hl, $306B
	ld (_pencol), hl
	push hl
	call UpdateClr
	pop hl
	ld (_pencol), hl
	ld a, (moves)
	call AtoDEC						;display number of moves made so far
	ld hl, $366B
	ld (_pencol), hl
	push hl
	call UpdateClr
	pop hl
	ld (_pencol), hl
	ld a, (cells)
	call AtoDEC						;display number of cells living
	ret

UpdateClr:
	ld a, ' '
	ld b, 10
UpdateClrLoop:							;clears space where number is displayed
	push af
	push bc
	call _vputmap
	pop bc
	pop af
	djnz UpdateClrLoop
	ret



;AtoDEC-- A to DECimal.  Displays number in a as decimal number up to 255.  All leading zeroes trimmed.
;by Cassady Roop
;------------------------------------------------------------------------------------------------------
AtoDEC:
    ld h, 0
	ld l, a
    ld b, 3
    ld de, num_spot+2
atodecmain:
      call $4044			;_HLdiv10   		
      add a, 48       		;convert to char code
      ld (de), a       		;store it to temp space
      dec de
      djnz atodecmain
      ld hl, num_spot
	xor a
	add a, 48			;char code for zero
	cp (hl)
	jr nz, atodecdisp
	inc hl
	cp (hl)
	jr nz, atodecdisp
	inc hl
atodecdisp:
      call _vputs     		;display it at current pen coords
      ret
num_spot:
	.db $00,$00,$00,0


;====================================================================
; DisplayHL:  (idea from SCaBBy/Mardell)    [Assembly Coder's Zenith]
;  Display HL as a 5-digit decimal number with no leading zeros
;  out: AF, HL, BC, DE destroyed
;====================================================================
DisplayHL:
 ld c,'0'  				; save ascii value for zero
 ld de,_OP1+5			; point to end of the buffer
 xor a				; zero terminate string
 ld (de),a				; set last byte to zero
DisplayHLl:
 call $4044			; next digit (_divHLby10)
 dec de				; next position
 add a,c				; convert to ascii
 ld (de),a				; save digit
 ld a,h				; load upper byte
 or l					; check with lower byte for zero
 jr nz,DisplayHLl			; loop
 ex de,hl				; point to buffer
 call _vputs			; print number
 ret					; we're done



;decodeABS   decodes absolute address in ahl.
;by Cassady Roop
;input -	AHL	= absolute address, like that outputted by _findsym
;output -	A	= page
;		HL	= offset into page.  Add $4000 for first swap window; $8000 for second swap window.
decodeABS:
	cp 0						;is a zero?
	ret z						;if so, hl is already a valid pointer
	dec a
	rlca						;*2
	rlca						;*4
	ld de, $4000				;length of one page
dABSloop:
	push hl					;save
	sbc hl, de					;hl=hl-de
	jr c, dABSdone				;if hl was less than $4000
	pop bc					;get rid of the value we saved
	inc a						;increment the page counter
	jr dABSloop					;reloop
dABSdone:
	inc a						;page correction
	pop hl					;the last pushed value is the offset
	ret						;we are done



;------------------------------------------------------------------------------
;TextIn version XmodX  modified version-- Text input that handles uppercase chars.
;by Cassady Roop
;INPUT: hl- points to string storage area.
;		b-  max number of characters to accept and store, up to 255.
;OUTPUT: string stored to desired location.
;		 string displayed at pen location on the screen.
;		 if LEFT is pressed, returns with nz. If successful, returns z.
;------------------------------------------------------------------------------
TextIn:
	push hl
	push bc
	call _getkey			;get keypress. it destroys hl.
	pop bc
	pop hl
	cp kenter
	jp z, textin_commit		;enter was pressed
	cp kleft				;LEFT?
	jr z, textin_cancel		;EXIT	
	ld c, a
	ld a, b					;load counter value
	cp 0					;see if char allowance has maxed out
	ret z					;too many characters, so return with Z.
	ld a, c
	push af					;preserve the keycode
	sub $1C					;k0
	jp m, TextIn_unknown	;it's not a known key if negative
	sub $0A					;k9 +1
	jp m, textin_unknown	;if neg, assumed to be a number key
	sub $02					;kCapA
	jp m, TextIn_unknown	;it's not a known key if negative
	sub $1A					;kCapZ +1
	jp m, textin_capletter	;if negative, then assume capital letter
	sub $1B					;kz +1
	jp m, textin_unknown	;if neg, assume lower case letter
textin_unknown:				;go here if it is unknown
	pop af					;what gets pushed, must get popped
	jr TextIn				;must be an unknown key at this point
textin_capletter:
	pop af
	add a, 25				;keycode-->char code
	ld (hl), a				;store
	call _vputmap			;display the letter
	inc hl					;increment string pointer
	dec b					;dec counter
	jr TextIn
textin_commit:				;enter key pressed
	xor a					;a=0
	cp 0					;set the z flag
	ret						;done
textin_cancel:				;exit key pressed
	xor a					;a=0
	cp 1					;res the z flag
	ret						;return with error




;simSprite   Cassady Roop   sprite putter of 6x8 (HxW) sprites to cells that are 1 byte wide and six rows tall.
;input- HL: sprite...   B: y grid coord, C: x grid coord.
;preserves HL (sprite) and DE and BC (coords)
simSprite:							;find screen byte by doing ($FC00+(y*$60)+x) | y=[b], x=[c]
	push bc							;this routine remembers the coords it was given
	push de							;this routine preserves [de]
	push hl							;save sprite location
	push hl							;save it again, so we can return it at the end
	ld hl, $0000					;clean slate
	ld de, $0060					;height of one grid row
	ld a, b							;get y value
	cp 0							;zero is a special case
	jr z, simSprite_found			;if so, then row is already found
simSprite_findloop:					;get row value by doing ([b]*$60) through additive multiplication
	add hl, de						;add $60
	djnz simSprite_findloop			;do that [b] times
simSprite_found:
	ld e, c							;put x coord into [e]
	add hl, de						;get offset into screen memory
	ld de, $FC00					;screen memory base address
	add hl, de						;add 'em to get cell address
	pop de							;retrieve sprite address
	ld bc, $0600					;loop six times
simSprite_draw:
	push bc							;save counter
	ld bc, $0010					;one screen row
	ld a, (de)						;get sprite byte
	ld (hl), a						;put byte into video memory
	add hl, bc						;drop down one screen row
	inc de							;next sprite byte
	pop bc							;remember counter
	djnz simSprite_draw				;loop [b] times
	pop hl							;this routine remembers sprite location in [hl]
	pop de							;this routine preserves [de]
	pop bc							;this routine remembers grid position
	ret




titletext:
	.db "Colony 1.0",0
credittext:
	.db "by Cassady Roop",0
credit2text:
	.db "SImmoralTech Software",0
helpthetext:
	.db "Help the ",0
growtext:
	.db " Grow!",0
addremovetext:
	.db "Add/Remove a cell",0
cellsdietext:
	.db "Cells die with ",0
neighborstext:
	.db "  neighbors",0
newcellstext:
	.db "New cells appear with ",0
secondtext:
	.db "[2ND]",0
movetext:
	.db "Move",0
f1text:
	.db "[F1]",0
helptext:
	.db "Help",0
leveltext:
	.db "Level",0
movestext:
	.db "Moves:",0
cellstext:
	.db "Cells:",0
goaltext:
	.db "Goal:",0
completetext:
	.db "Complete",0
colonytext:
	.db "Colony",0
hasdiedtext:
	.db "Has Died",0
haveatext:
	.db "You have a ",0
hightext:
	.db "High ",0
scoretext:
	.db "Score",0
enterinitialstext:
	.db "Enter Initials: ",0
pointstext:
	.db "Points: ",0

progname:
	.db 6,"Colony"
highscore:							;this is the highscore storage word and initials written back to the program var
	.dw $0000
	.db "CSR",0						;initials, mine, with starting score of zero.						


cellSprite:							;an active cell
	.db %11111111
	.db %10000000
	.db %10111110
	.db %10110110
	.db %10111110
	.db %10000000

openSprite:							;an open cell
	.db %11111111
	.db %10000000
	.db %10000000
	.db %10000000
	.db %10000000
	.db %10000000


cursorOpenSprite:						;cursor used over an open square
	.db %11111111
	.db %11111111
	.db %11111111
	.db %11111111
	.db %11111111
	.db %11111111

cursorCellSprite:						;cursor to use over an occupied square
	.db %11111111
	.db %11111111
	.db %11000001
	.db %11001001
	.db %11000001
	.db %11111111

arrowSprite:
	.db %11100001, %00001110
	.db %10000011, %10000010
	.db %10000111, %11000010
	.db %00000001, %00000000
	.db %00000001, %00000000
	.db %00100001, %00001000
	.db %01100000, %00001100
	.db %11111101, %01111110
	.db %01100000, %00001100
	.db %00100001, %00001000
	.db %00000001, %00000000
	.db %00000001, %00000000
	.db %10000111, %11000010
	.db %10000011, %10000010
	.db %11100001, %00001110



;level format
; .db [number of cells needed to advance]
; .db [number of cells]
; .db string of 1-byte offsets denoting entry points in the array.  Must be in order!

LevelIndex:
	.dw Level1,Level2,Level3,Level4,Level5,Level6,Level7,Level8,Level9,Level10,Level11
	.dw Level12,Level13,Level14,Level15,Level16,Level17,Level18,Level19,Level20


Level1:
	.db 5
	.db 3
	.db 0,1,2

Level2:
	.db 7
	.db 5
	.db 0,25,40,42,43

Level3:
	.db 8
	.db 5
	.db 10,11,20,22,99

Level4:
	.db 10
	.db 6
	.db 0,3,5,6,16,10

Level5:
	.db 15
	.db 6
	.db 0,2,3,10,20,25

Level6:
	.db 20
	.db 7
	.db 22,25,26,33,44,55,62

Level7:
	.db 25
	.db 4
	.db 23,25,43,45

Level8:
	.db 25
	.db 5
	.db 34,35,36,37,45

Level9:
	.db 30
	.db 10
	.db 23,24,25,26,34,53,54,55,56,64

Level10:
	.db 35
	.db 60
	.db 0,1,2,3,4,5,6,7,8,9,10,19,20,29,30,39,40,49,50,59,60,69,70,79,80,89,90,91,92,93,94,95,96,97,98,99
	.db 21,31,41,51,61,71,28,38,48,58,68,78,12,13,14,15,16,17,82,83,84,85,86,87

Level11:
	.db 44
	.db 36
	.db 22,23,24,25,26,27,32,33,34,35,36,37,42,43,44,45,46,47,52,53,54,55,56,57,62,63,64,65,66,67,72,73,74
	.db 75,76,77

Level12:
	.db 28
	.db 8
	.db 0,11,9,18,88,99,90,81
Level13:
	.db 30
	.db 4
	.db 35,44,47,56
Level14:
	.db 32
	.db 18
	.db 9,8,18,17,27,26,36,35,45,44,54,53,63,62,72,71,81,80
Level15:
	.db 40
	.db 20
	.db 4,5,13,16,22,27,31,38,40,49,50,59,61,68,72,77,83,86,94,95
Level16:
	.db 45
	.db 20
	.db 0,9,11,18,22,27,33,36,44,45,54,55,63,66,72,77,81,88,90,99
Level17:
	.db 35
	.db 25
	.db 0,1,10,4,5,10,14,21,25,30,31,34,35,64,65,68,69,74,78,85,89,94,95,98,99
Level18:
	.db 33
	.db 18
	.db 1,11,17,18,19,21,34,35,36,47,51,57,61,64,67,71,74,84
Level19:
	.db 33
	.db 10
	.db 15,24,25,26,35,44,53,54,55,64
Level20:
	.db 45
	.db 10
	.db 40,41,42,43,44,45,46,47,48,49





.end
