; Worms!
;
; Version 1.0
;
; By David Hart 1998
;
; Includes
	#include "asm86.h"
	#include "ti86asm.inc"
	#include "ti86math.inc"
	#include "ti86ops.inc"

	.org	_asm_exec_ram

; ***** Constant equates
NUMBER_OPTIONS	equ	3	; Number of entries in options menu
DEF_NUM_TEAMS	equ	2	; Default number of teams
DEF_NUM_WORMS	equ	2	; Default number of worms per team
AIM_DISTANCE	equ	15	; Distance of aim sprite from worm
TURN_TIME	equ	60	; Initial turn time (seconds)
NUM_WEAPONS	equ	5	; Number of different weapons

; ***** Timing values (number of times interrupt must be called)
TIMER_DELAY	equ	190	; Time before timer at top of screen
				; is incremented
MESSAGE_DELAY	equ	500	; Time before message disappears
KEY_REP_DELAY	equ	150	; Key repeat speed
SLOW_KEY_REP_DELAY	equ	2	; Slow key repeat speed (on top of
					; normal key delay)
FLASH_ACTIVE_DELAY	equ	120	; Speed at which active worm
				; indicator flashes

; ***** Memory equates
disp_page_1	equ	$3C	; Graphic viewport 1 (+$C0 equal MSB
				; of address)
disp_page_2	equ	$30	; Graphic viewport 2

; ***** Data variables
cur_page	equ	_textShadow	; Current graphics page
					; (disp_page_1 or 2) (1 BYTE)

; These two MUST be together, and in this order
num_worms	equ	_textShadow+1	; Number of worms per team
					; (1 BYTE)
num_teams	equ	_textShadow+2	; Number of worm teams (1 BYTE)

x_map		equ	_textShadow+3	; Location of x-coord of top
				; left hand corner of visible map
				; (1 BYTE) - in blocks of 8-pixels
y_map		equ	_textShadow+4	; Location of y-coord of top
				; left hand corner of visible map
				; (1 BYTE) - in blocks of 8-pixels
key_rep		equ	_textShadow+5	; Value used to determine key
				; repeat speed (2 BYTES)
rand_seed	equ	_textShadow+7	; Random seed value (2 BYTES)
temp_word	equ	_textShadow+9	; Temporary word (2 BYTES)
total_num_worms	equ	_textShadow+11	; Total no. of worms (1 BYTE)
timer_count	equ	_textShadow+12	; Counter for visible timer
					; (2 BYTES)
turn_time_left	equ	_textShadow+14	; Time left in seconds (1 BYTE)
mode		equ	_textShadow+15	; Current game mode (1 BYTE)
interrupt_byte	equ	_textShadow+16	; Used by user-interrupt -
				; Set bit 0 to update display (1 BYTE)
cur_team	equ	_textShadow+17	; Number of currently active
					; team (1 BYTE)
cur_worm	equ	_textShadow+18	; Number of currently active
					; worm out of all worms, not
					; just current team (1 BYTE)
message_x	equ	_textShadow+19	; X pos of message (1 BYTE)
message_timer	equ	_textShadow+20	; Counter for time message
					; displays before disappearing
					; (2 BYTES)
int_redraw_check	equ	_textShadow+22	; Used by interrupt to
				; check whether it needs to redraw the
				; display (1 BYTE)
temp_byte	equ	_textShadow+23	; Temporary byte (1 BYTE)
target_x	equ	_textShadow+24	; Target X screen location when
					; moving screen (1 BYTE)
target_y	equ	_textShadow+25	; Target Y screen location when
					; moving screen (1 BYTE)
active_flash_timer	equ	_textShadow+26	; Counter for flashing
				; active worm indicator (2 BYTES)
active_flash_on	equ	_textShadow+28	; First bit set when active
				; worm indicator is on (1 BYTE)
worm_aim_direction	equ	_textShadow+29	; Direction worm is
				; aiming in: 0 to 180 degrees (1 BYTE)

; Projectile variables (displacement variables also used for aim sprite)
;  - LSB of displacement is "after the decimal point"
;  - Values are stored in straight byte order, from MSB to LSB
x_vel		equ	_textShadow+30	; X velocity (3 BYTES)
y_vel		equ	_textShadow+33	; Y velocity (2 BYTES)
x_dis		equ	_textShadow+35	; X displacement (3 BYTES)
y_dis		equ	_textShadow+38	; Y displacement (2 BYTES)
initial_vel	equ	_textShadow+40	; Initial velocity (1 BYTE)
wind		equ	_textShadow+41	; X deceleration - wind effect
					; (1 BYTE)

temp_worm_data1	equ	_textShadow+42	; Temporary worm data - used by
					; map_redraw (1 BYTE)
cur_weapon_select	equ	_textShadow+43	; Currently selected weapon
						; (1 BYTE)
must_release_keys	equ	_textShadow+44	; Bits set when indicated
				; keys are being held down - key must be
				; released before it will respond again
				; Bits:
				;	0 = MORE key - Weapon select menu up
				;	1 = ENTER key - Select weapon/fire
				; (1 BYTE)
slow_key_rep		equ	_textShadow+45	; Value used to determine key
				; repeat speed - for slowly repeating keys
				; (1 BYTE)

; ***** Game modes (values indicate the bit numbers that are set)
m_worm_playing	equ	0		; Worm is currently being
					; moved, waiting or aiming
m_showing_message	equ	1	; Currently showing a message
m_firing	equ	2		; Currently firing weapon,
					; and tracking projectile
m_finding_worm	equ	3		; Currently moving across map
					; to newly active worm
m_weapon_select	equ	4		; Weapon select menu is currently
					; open
m_fixed_message	equ	5		; Currently showing a fixed message
					; that will not disappear after a
					; set amount of time
m_charging_weapon	equ	6	; Currently charging weapon (finding
					; force to fire it with

; ***** Map generating equates
gen_map1	equ	_plotSScreen		; Size 12 bytes
gen_map2	equ	_plotSScreen+12		; Size 48 bytes
gen_map3	equ	_plotSScreen+60		; Size 192 bytes
gen_map4	equ	_plotSScreen+252	; Size 768 bytes

; ***** Worm equates
worm_data	equ	_plotSScreen	; Max 252 bytes (31 worms
					; of 8 bytes each)
team_data	equ	_plotSScreen+252	; Info on each team
					; (CAN ONLY BE USED AFTER
					; worm_data HAS BEEN SET UP)

; ***** Miscellaneous equates
temp_data	equ	_cmdShadow

; ***** Worm data offsets
w_health	equ	0	; 8 bits

w_team		equ	1	; 3 bits
w_member	equ	1	; 2 bits
w_anim_cell	equ	1	; 3 bits

w_direction	equ	2	; 4 bits (1 per direction)
w_weapon	equ	2	; 4 bits

w_action	equ	3	; 3 bits
w_speed		equ	3	; 4 bits

w_x_worm	equ	3	; 9 bits

w_y_worm	equ	5	; 8 bits

w_name		equ	6	; 8 bits

w_reserved	equ	7	; 8 bits

; ***** Worm action mask values
a_stopped	equ	%00000000	; Worm is stopped
a_moving	equ	%00100000	; Worm is moving
a_jumping	equ	%01000000	; Worm is jumping
a_aiming	equ	%01100000	; Worm is aiming weapon
a_firing	equ	%10000000	; Worm is firing weapon
a_thrown	equ	%10100000	; Worm is thrown by explosion

; ***** Worm direction mask values
d_left		equ	%10000000
d_right		equ	%01000000
d_up		equ	%00100000
d_down		equ	%00010000

; ***** Team data offsets (1 byte each)
t_num_worms	equ	0	; Number of worms alive in team

; ***** Paged RAM space equates
actual_map	equ	$8800	; The actual storage area for the map
worm_name_table	equ	$87C0	; Table of two byte pointers to worm
				; names (64 bytes)

paged_RAM_start	equ	$87C0	; Start of used paged RAM
used_paged_RAM	equ	$3040	; Amount of paged RAM used

; ***** Other addresses
message_sprite	equ	$CD1A	; Buffer for text messages
value_buffer	equ	$CD8A	; Buffer for storage of sprite for
				; turn time, health, etc.
value_sprite1	equ	$CCFC	; 6 X 2 byte sprites for numeric data
value_sprite2	equ	$CD03	; 6 X 2 byte sprites for numeric data
_user_int_ram	equ	$D2FD	; Area where interrupt handler goes

; ***** Main Program

; Setup
	call	_runindicoff	; Standard setup calls
	call	MENUS_OFF
	call	_clrLCD
	res	appTextSave,(iy+appflags)
	ld	a,disp_page_1
	ld	(cur_page),a

	ld	a,DEF_NUM_TEAMS		; Setup variables
	ld	(num_teams),a
	ld	a,DEF_NUM_WORMS
	ld	(num_worms),a
	xor	a
	ld	(cur_worm),a
	ld	(cur_team),a
	ld	a,r
	ld	(rand_seed+1),a
	ld	a,0
	ld	(mode),a

; Setup and enable user-interrupt routine for handling the display
        res     2,(iy+$23)

        ld      hl,IntRoutine
        ld      de,_user_int_ram+1
        ld      bc,InterruptMemEnd-IntRoutine
        ldir

        ld      a,(_user_int_ram+1)
        ld      hl,_user_int_ram+($28*1)
        add     a,(hl)
        ld      hl,_user_int_ram+($28*2)
        add     a,(hl)
        ld      hl,_user_int_ram+($28*3)
        add     a,(hl)
        ld      hl,_user_int_ram+($28*4)
        add     a,(hl)
        ld      hl,_user_int_ram+($28*5)
        add     a,(hl)
        ld      (_user_int_ram),a

	xor	a			; Setup interrupt variables
	ld	(interrupt_byte),a

; Main procedures
	call	showOptions	; Show initial options screen
	set	textwrite,(iy+new_grf_flgs)
	set	trigdeg,(iy+trigflags)
	call	playGame	; Run the actual game

; Clear up
	res	textwrite,(iy+new_grf_flgs)
	res     2,(iy+$23)	; Disable user-interrupt

	ld	a,$20		; Clear text shadow
	ld	b,168
	ld	hl,_textShadow
	call	fill_8_bit

	ld	b,168		; Clear command shadow
	ld	hl,_cmdShadow
	call	fill_8_bit

	ld	hl,paged_RAM_start	; Clear used "free exec RAM"
	xor	a
	ld	(hl),a
	ld	de,paged_RAM_start+1
	ld	bc,used_paged_RAM-1
	ldir

	ld	a,$3C
	out	(0),a
	call	_clrLCD
	call	_runindicon
	ret

; ***** Show options screen, and handle keyboard input
showOptions:
	ld	c,NUMBER_OPTIONS	; Current highlighted option
				; REVERSED, i.e. first option =
				; NUMBER_OPTIONS

; Draw the title bar
redraw_options_menu:
	push	bc
	call	clear_current_page
	pop	bc
	ld	a,(cur_page)	; Turn top of screen black
	add	a,$C0
	ld	h,a
	xor	a
	ld	l,a
	dec	a
	ld	b,112
	call	fill_8_bit

	set	textInverse,(iy+textflags)	; Show title
	ld	a,1
	ld	(_penCol),a
	xor	a
	ld	(_penRow),a
	ld	hl,title_bar
	call	_vputs
	res	textInverse,(iy+textflags)

; Show the list of options
	ld	a,4
	ld	(_penRow),a
	ld	b,NUMBER_OPTIONS
	ld	hl,options_list
draw_option:
	ld	a,(_penRow)
	add	a,6
	ld	(_penRow),a
	ld	a,20
	ld	(_penCol),a
	ld	a,b
	cp	c
	jr	nz,do_draw_option
	set	textInverse,(iy+textflags)
do_draw_option:
	call	_vputs				; Show the option

	push	hl
	ld	a,b			; Show numerical values if
	cp	NUMBER_OPTIONS-1	; needed
	jr	nz,check_num_worms_option
	ld	a,(num_teams)
	ld	l,a
	ld	h,0
	call	V_D_HL_DECI

check_num_worms_option:
	cp	NUMBER_OPTIONS-2
	jr	nz,another_option
	ld	a,(num_worms)
	ld	l,a
	ld	h,0
	call	V_D_HL_DECI

another_option:
	pop	hl
	res	textInverse,(iy+textflags)
	djnz	draw_option

; Get key input, and process
check_option_key_press:
	call	_getkey
	cp	kUp
	jr	z,option_up
	cp	kDown
	jr	z,option_down
	cp	kEnter
	jr	z,option_enter
	jr	check_option_key_press

option_up:
	ld	a,NUMBER_OPTIONS
	cp	c
	jr	z,check_option_key_press
	inc	c
	jp	redraw_options_menu

option_down:
	ld	a,1
	cp	c
	jr	z,check_option_key_press
	dec	c
	jp	redraw_options_menu

option_enter:
	ld	a,r		; Get LSB of random seed
	ld	(rand_seed),a

	ret

; ***** Run the actual game
playGame:

; Initial setting up
	call	_clrLCD		; Clear the screen
	ld	hl,TIMER_DELAY
	ld	(timer_count),hl
redo_setup_map:
	call	setupMap
	call	setupWorms
	jr	c,redo_setup_map	; If not enough space for all
					; worms, draw another map

; Setup to move to currently active worm initially
	ld	hl,FLASH_ACTIVE_DELAY
	ld	(active_flash_timer),hl
	call	set_next_turn
        set     2,(iy+$23)		; Enable interrupt
	call	redraw_map
	halt
	call	redraw_map

; Handle key presses (and other real-time events)
handle_key:

; ************** DO REAL-TIME UPDATING DEPENDING ON MODE **************

; Check for "move display to focus on current worm" option
	ld	a,(mode)
	bit	m_finding_worm,a
	jr	z,not_moving_display_to_worm
	ld	c,0			; Check whether x-pos of screen
	ld	a,(target_x)		; needs to be moved
	ld	b,a
	ld	a,(x_map)
	cp	b
	jr	z,no_change_x_disp
	jr	c,do_inc_x_disp
	dec	a
	ld	c,1
	jr	no_change_x_disp
do_inc_x_disp:
	inc	a
	ld	c,1

no_change_x_disp:
	ld	(x_map),a
	ld	a,(target_y)		; Check whether y-pos of screen
	ld	b,a			; needs to be moved
	ld	a,(y_map)
	cp	b
	jr	z,no_change_y_disp
	jr	c,do_inc_y_disp
	dec	a
	ld	c,1
	jr	no_change_y_disp
do_inc_y_disp:
	inc	a
	ld	c,1

no_change_y_disp:
	ld	(y_map),a
	push	bc
	call	redraw_map
	pop	bc

	xor	a
	cp	c
	jr	nz,not_moving_display_to_worm
	ld	a,(mode)
	res	m_finding_worm,a
	ld	(mode),a

not_moving_display_to_worm:

; Check for "timed-out" timer, forcing end of turn
	ld	a,(turn_time_left)
	cp	0
	call	z,turn_timed_out

; *********************************************************************

; Actually get keys
	ld	a,(must_release_keys)	; Keys that cannot be held down to
	ld	b,a		; keep repeating - info in B
	call	q_get_key
	cp	%01100110	; Check for EXIT key
	jp	z,handle_EXIT

; Only check these keys if key-repeat timer has timed out
	ld	hl,(key_rep)
	ld	de,0
	or	a
	sbc	hl,de
	jr	z,check_timed_keys
	dec	hl
	ld	(key_rep),hl
	jp	handle_key

check_timed_keys:
	cp	%00110001	; Check for DOWN key (2)
	jr	z,handle_DOWN
	cp	%01000010	; Check for LEFT key (4)
	jr	z,handle_LEFT
	cp	%00100010	; Check for RIGHT key (6)
	jr	z,handle_RIGHT
	cp	%00110011	; Check for UP key (8)
	jr	z,handle_UP
	cp	%00110010	; Check for CENTRE ON WORM (5)
	jr	z,handle_CENTRE
	cp	%00000011	; Check for AIM UP key
	jr	z,handle_AIM_UP
	cp	%00000000	; Check for AIM DOWN key
	jp	z,handle_AIM_DOWN
	cp	%00010000	; Check for FIRE key (ENTER)
	jp	z,handle_FIRE
	res	1,b
	cp	%01100111	; Check for VIEW WEAPONS key (MORE)
	jp	z,handle_VIEW_WEAPONS
	res	0,b
	cp	%00000001	; Check for MOVE LEFT key
	jp	z,handle_MOVE_LEFT
	cp	%00000010	; Check for MOVE RIGHT key
	jp	z,handle_MOVE_RIGHT
	ld	a,b
	ld	(must_release_keys),a

	jp	handle_key

handle_UP:
	ld	a,(y_map)
	cp	0
	jp	z,handle_key
	dec	a
	ld	(y_map),a
	call	redraw_map
	jp	handle_key_repeat

handle_DOWN:
	ld	a,(y_map)
	cp	24
	jp	z,handle_key
	inc	a
	ld	(y_map),a
	call	redraw_map
	jp	handle_key_repeat

handle_LEFT:
	ld	a,(x_map)
	cp	0
	jp	z,handle_key
	dec	a
	ld	(x_map),a
	call	redraw_map
	jp	handle_key_repeat

handle_RIGHT:
	ld	a,(x_map)
	cp	32
	jp	z,handle_key
	inc	a
	ld	(x_map),a
	call	redraw_map
	jp	handle_key_repeat

handle_CENTRE:
	call	set_move_to_current_worm
	jp	handle_key_repeat

handle_AIM_UP:
	ld	a,(worm_aim_direction)
	cp	0
	jp	z,handle_key
	sub	10
	ld	(worm_aim_direction),a
	ld	b,a_aiming
	call	set_current_worm_action
	call	get_aim_sprite_coords
	call	redraw_map
	jp	handle_key_repeat

handle_AIM_DOWN:
	ld	a,(worm_aim_direction)
	cp	180
	jp	z,handle_key
	add	a,10
	ld	(worm_aim_direction),a
	ld	b,a_aiming
	call	set_current_worm_action
	call	get_aim_sprite_coords
	call	redraw_map
	jp	handle_key_repeat

handle_FIRE:
	ld	a,(must_release_keys)
	bit	1,a
	jp	nz,handle_key_repeat
	set	1,a
	ld	(must_release_keys),a

	ld	a,(mode)
	bit	m_weapon_select,a
	jr	nz,handle_SELECT_WEAPON

	;call	do_firing		***** DOESN'T WORK
	call	set_next_worm		; } Called instead of do_firing
	call	set_next_turn		; }

	jp	handle_key_repeat
handle_SELECT_WEAPON:
	res	m_weapon_select,a
	res	m_fixed_message,a
	ld	(mode),a
	call	redraw_map
	jp	handle_key_repeat

handle_VIEW_WEAPONS:
	ld	a,(must_release_keys)
	bit	0,a
	jr	nz,handle_key_repeat
	set	0,a
	ld	(must_release_keys),a

	ld	a,(mode)
	bit	m_weapon_select,a
	jr	z,set_weapon_select_on
	res	m_weapon_select,a
	res	m_fixed_message,a
	jr	update_weapon_select_mode
set_weapon_select_on:
	set	m_weapon_select,a
update_weapon_select_mode:
	ld	(mode),a
	call	redraw_map
	jr	handle_key_repeat

handle_MOVE_LEFT:
	ld	a,(mode)
	bit	m_weapon_select,a
	jr	nz,handle_WEAPON_SELECT_LEFT
	jr	handle_key_repeat
handle_WEAPON_SELECT_LEFT:
	ld	a,(slow_key_rep)
	cp	0
	jr	z,weapon_select_left_enabled
	dec	a
	ld	(slow_key_rep),a
	jr	handle_key_repeat
weapon_select_left_enabled:
	ld	a,SLOW_KEY_REP_DELAY
	ld	(slow_key_rep),a

	ld	a,(cur_weapon_select)
	cp	0
	jr	z,handle_key_repeat
	dec	a
	ld	(cur_weapon_select),a
	call	redraw_map
	jr	handle_key_repeat

handle_MOVE_RIGHT:
	ld	a,(mode)
	bit	m_weapon_select,a
	jr	nz,handle_WEAPON_SELECT_RIGHT
	jr	handle_key_repeat
handle_WEAPON_SELECT_RIGHT:
	ld	a,(slow_key_rep)
	cp	0
	jr	z,weapon_select_right_enabled
	dec	a
	ld	(slow_key_rep),a
	jr	handle_key_repeat
weapon_select_right_enabled:
	ld	a,SLOW_KEY_REP_DELAY
	ld	(slow_key_rep),a

	ld	a,(cur_weapon_select)
	cp	NUM_WEAPONS-1
	jr	z,handle_key_repeat
	inc	a
	ld	(cur_weapon_select),a
	call	redraw_map
	jr	handle_key_repeat

handle_key_repeat:
	ld	hl,KEY_REP_DELAY
	ld	(key_rep),hl
	jp	handle_key

handle_EXIT:
	ret

; ***** Setup the map
setupMap:
	ld	hl,paged_RAM_start	; Clear map
	xor	a
	ld	(hl),a
	ld	de,paged_RAM_start+1
	ld	bc,used_paged_RAM-1
	ldir

; Build up map in graph memory - start with very blocky map, filled
; randomly wiht 0's and 1's
	ld	b,48
	ld	hl,gen_map2
fill_first_map:
	call	get_rand_bit
	ld	(hl),a
	inc	hl
	djnz	fill_first_map

; Double map dimensions, and copy existing map to new map
	ld	hl,gen_map3
	ld	de,gen_map2
	ld	b,48
	ld	c,12-2
	call	enlarge_map

; Randomly add or remove blocks
	ld	b,4
	ld	hl,gen_map3
	ld	de,192
	ld	c,12
randomize_map_3:
	call	random_build_map
	call	random_build_map_side
	djnz	randomize_map_3

; Double map dimensions, and copy existing map to new map
	ld	hl,gen_map4
	ld	de,gen_map3
	ld	b,192
	ld	c,24-2
	call	enlarge_map

; Randomly add or remove blocks
	ld	b,10
	ld	hl,gen_map4
	ld	de,768
	ld	c,24
randomize_map_4:
	call	random_build_map
	call	random_build_map_side
	djnz	randomize_map_4

; Draw border around final map
	ld	b,24
	ld	hl,gen_map4
	ld	a,$01
fill_top_border:
	ld	(hl),a
	inc	hl
	djnz	fill_top_border

	ld	b,24
	ld	hl,gen_map4+744
	ld	a,$01
fill_bottom_border:
	ld	(hl),a
	inc	hl
	djnz	fill_bottom_border

	ld	hl,gen_map4+23
	ld	de,23
	ld	b,31
	ld	a,$01
fill_side_borders:
	ld	(hl),a
	inc	hl
	ld	(hl),a
	add	hl,de
	djnz	fill_side_borders

; Copy generated map to main map background
	call	enlarge_to_background

	ret

; ***** Setup worm data
setupWorms:

; Find positions for all worms (store in temporary memory area)
	ld	bc,(num_worms)	; Worms per team in C, no. teams in B
	xor	a
	ld	(temp_byte),a	; Temporary byte holds worm name number
calc_total_num_worms:		; Find total number of worms
	add	a,c
	djnz	calc_total_num_worms
	ld	(total_num_worms),a

	ld	hl,gen_map4-1	; Address of map to scan
	ld	de,temp_data	; Where to store worm pos data
	ld	bc,$17ff	; B used to hold x block, C - y block
find_suitable_pos:
	push	hl
	push	de
	ex	de,hl
	ld	hl,gen_map4+767
	or	a		; Test for end of map reached, and
	sbc	hl,de		; return with carry flag set if so
	pop	de
	pop	hl
	ret	c

	push	af		; Save total number of worms on stack
	push	de
	push	bc
find_blank_block:
	pop	bc
	inc	b
	ld	a,24
	cp	b
	jr	nz,check_block_empty
	ld	b,0
	inc	c

check_block_empty:
	push	bc
	inc	hl		; Find an empty block
	ld	a,(hl)
	cp	0	
	jr	nz,find_blank_block

	push	hl
	ld	de,24		; Get value of block below in B, and
	add	hl,de		; block below that in C
	ld	b,(hl)
	add	hl,de
	ld	c,(hl)
	pop	hl

	xor	a		; Check block immediately below is
	cp	b		; clear, and block below that is set
	jr	nz,find_blank_block
	cp	c
	jr	z,find_blank_block

	pop	bc
	pop	de		; Acceptable position for worm found,
	ex	de,hl		; so set position
	ld	(hl),b
	inc	hl
	ld	(hl),c
	inc	hl
	ex	de,hl
	push	bc
	ld	bc,24
	add	hl,bc
	pop	bc
	inc	c

	pop	af
	dec	a
	jr	nz,find_suitable_pos

; temp_data now contains a table, each entry two bytes, containing
; block locations for worm positions.  Now, with that information,
; set up each worms data
	ld	a,(total_num_worms)
	ld	b,a		; Total no. worms in B
	ld	hl,0		; Team number in H, member number in L
	ld	(temp_word),hl
	ld	hl,worm_data
	ld	de,temp_data

add_worm:
	ld	a,100
	ld	(hl),a		; Health to 100

	inc	hl
	push	de
	ld	de,(temp_word)
	ld	a,d
	rlca
	rlca
	rlca
	rlca
	rlca
	ld	(hl),a		; Adjust team number to store as worm
	ld	c,a		; data
	ld	a,e
	rlca
	rlca
	rlca
	or	c		; Adjust team member number to store
	ld	(hl),a		; as worm data

	inc	d		; Update team and member numbers
	ld	a,(num_teams)
	cp	d
	jr	nz,no_change_member_number
	ld	d,0
	inc	e

no_change_member_number:
	ld	(temp_word),de
	inc	hl
	ld	a,d_left	; Direction to "left"
	ld	(hl),a		; Weapon, action, speed to
	inc	hl		; zero (latter two done with x-pos)

	pop	de
	ld	a,(de)		; Get X pos and multiply by 16 to get
	inc	de		; pixel position
	rlca
	rlca
	rlca
	rlca
	push	af
	ld	a,0
	rla
	ld	(hl),a		; Set action, speed to zero, and set
	inc	hl		; MS bit of X-pos to correct value
	pop	af
	ld	(hl),a		; Set LSB of x-pos
	inc	hl

	ld	a,(de)		; Get Y pos and multiply by 8 to get
	inc	de		; pixel position
	rlca
	rlca
	rlca
	and	%11111000
	ld	(hl),a

	inc	hl		; Set name (number of name in table)
	ld	a,(temp_byte)
	ld	(hl),a
	inc	a
	ld	(temp_byte),a

	xor	a
	inc	hl
	ld	(hl),a		; Reserved byte to zero
	inc	hl

	djnz	add_worm

; Setup teams
	ld	a,(num_teams)
	ld	b,a
	ld	hl,team_data

add_team:
	ld	a,(num_worms)
	ld	(hl),a
	inc	hl
	djnz	add_team

; Setup worm name table
	ld	de,actual_worm_names
	ld	hl,worm_name_table
	ld	a,(total_num_worms)
	ld	b,a
add_worm_name:
	ld	(hl),e		; Fill table with pointers
	inc	hl
	ld	(hl),d
	inc	hl
next_worm_name:
	inc	de
	ld	a,(de)
	cp	0
	jr	nz,next_worm_name
	inc	de
	djnz	add_worm_name

	or	a	; Clear carry flag (indicates success)
	ret

; ***** Redraws the visible area of the map
redraw_map:
	di
	ld	a,(y_map)	; Find byte for top left of screen
	ld	h,0		; (multiply y_map by 40)
	ld	l,a
	add	hl,hl
	add	hl,hl
	add	hl,hl
	add	hl,hl
	ld	b,h
	ld	c,l
	add	hl,hl
	add	hl,bc

	add	hl,hl		; Now multiply by another 8 to skip
	add	hl,hl		; 8 pixels rather than 1
	add	hl,hl

	ld	a,(x_map)	; Add on x-coord
	ld	b,0
	ld	c,a
	add	hl,bc
	ld	bc,actual_map
	add	hl,bc

; Copy map to invisible viewport
	.db	$11		; LD DE,viewport address
	.db	$00
redraw_map_write_addr:
	.db	$FC
	
	ld	a,64
draw_screen_line:
	ld	bc,16
	ldir
	ld	bc,32
	add	hl,bc
	dec	a
	jr	nz,draw_screen_line

; Draw all visible worms
	ld	a,(total_num_worms)
	ld	b,a
	ld	hl,worm_data

draw_worm:
	push	hl
	ld	a,(hl)
	cp	0		; Only draw if health greater than
	jp	z,no_draw_worm	; zero
	inc	hl
	ld	a,(hl)
	and	%00000111	; Mask off everything but animation
	ld	c,a		; cell
	inc	hl
	inc	hl

	ld	a,(hl)
	ld	(temp_worm_data1),a	; Store for later
	and	%00000001	; Mask all but MS bit of x pos
	ld	d,a
	inc	hl
	ld	a,(hl)		; Get rest of x pos (all now in DE)
	ld	e,a

	push	hl
	ld	a,(x_map)
	call	mul_a_by_8
	ex	de,hl
	or	a		; Subtract x-pos of view screen from
	sbc	hl,de		; x-pos to get screen position
	ex	de,hl
	pop	hl

	jp	c,no_draw_worm	; Don't draw if off left of screen
	push	de
	push	hl
	ld	hl,$7F	
	or	a
	sbc	hl,de
	pop	hl
	pop	de
	jp	c,no_draw_worm	; Don't draw if off right of screen
	push	de		; Store x-pos on stack

	inc	hl
	ld	a,(hl)		; Get y pos (all now in DE)
	ld	e,a
	ld	d,0

	push	hl
	ld	a,(y_map)
	call	mul_a_by_8
	ex	de,hl
	or	a		; Subtract y-pos of view screen from
	sbc	hl,de		; y-pos to get screen position
	ex	de,hl
	pop	hl

	jr	c,ext_check	; Don't draw if off top of screen
	ld	a,e
	pop	de
	cp	64
	jr	nc,no_draw_worm	; Don't draw if off bottom of screen
	jr	cont_draw_worm

ext_check:
	ld	a,$FF
	cp	d
	jr	nz,ext_check_fail
	ld	a,$F7
	cp	e
	jr	nc,ext_check_fail
	ld	a,e
	pop	de
	jr	cont_draw_worm
ext_check_fail:
	pop	de
	jr	no_draw_worm

cont_draw_worm:
	ld	d,a		; x-pos in E, y-pos in D
	ld	a,4
	add	a,d
	ld	d,a
	ld	a,c
	push	de
	push	bc
	call	PutWorm
	pop	bc
	pop	de

	ld	a,(temp_worm_data1)	; Recall worm's action
	and	%11100000
	cp	a_aiming
	jr	z,draw_worm_aiming
	jr	check_if_active
draw_worm_aiming:
	push	de
	push	bc
	ld	a,(y_dis)
	add	a,d
	add	a,2
	ld	(y_vel),a
	ld	d,a
	ld	a,(x_dis)
	add	a,e
	ld	(x_vel),a
	ld	e,a
	ld	hl,aim_sprite
	call	PutSprite
	pop	bc
	pop	de
	jr	check_if_active

check_if_active:
	ld	a,(total_num_worms)	; Check if current worm is
	sub	b			; active
	ld	c,a
	ld	a,(cur_worm)
	cp	c
	jr	nz,no_draw_worm

	ld	a,(active_flash_on)
	cp	0
	jr	z,no_draw_worm
	push	bc			; Worm is active, so draw an
	ld	a,d			; active indicator above it
	ld	d,6
	sub	d
	ld	d,a
	ld	hl,active_sprite
	call	PutSprite
	pop	bc

no_draw_worm:
	ld	de,8
	pop	hl
	add	hl,de
	dec	b
	jp	nz,draw_worm

; Show the turn time remaining
	ld	a,(turn_time_left)
	call	writeNumericValue
	ld	hl,value_sprite1
	ld	d,1
	ld	e,58
	call	PutSprite
	ld	hl,value_sprite2
	ld	d,1
	ld	e,66
	call	PutSprite

; Show weapon select menu if required
	ld	a,(mode)
	bit	m_weapon_select,a
	jr	z,no_show_weapon_select
	call	show_weapon_select
no_show_weapon_select:

; Show a message if required
	ld	a,(mode)
	bit	m_showing_message,a
	jr	z,no_show_message
	call	show_message
no_show_message:

; Show a fixed message if required
	ld	a,(mode)
	bit	m_fixed_message,a
	jr	z,no_show_fixed_message
	call	show_message
no_show_fixed_message:

; Show "weapon charging" bar if required
	ld	 a,(mode)
	bit	m_charging_weapon,a
	jr	z,not_charging
	call	show_weapon_charging_bar
not_charging:

; Track a projectile if required
	ld	a,(mode)
	bit	m_firing,a
	jr	z,not_firing
	call	track_projectile
not_firing:

	ld	a,(interrupt_byte)	; Set display to be switched
	set	0,a			; at next VBL interrupt
	ld	(interrupt_byte),a
	ei
	ret

; ***** Enlarges map (address in DE) to double it's dimensions, and
;	stores in memory pointed to by HL.  Modifies all registers.
;	X dimension-2 of new map in C, total size of old map in B.
enlarge_map:
	push	hl		; Store address of end of line
	push	bc
	ld	b,0
	add	hl,bc
	inc	hl
	inc	hl
	ld	(temp_word),hl
	pop	bc
	pop	hl

enlarge_line:
	ld	a,(de)		; Load top of block of new map
	push	de
	ld	(hl),a
	inc	hl
	ld	(hl),a
	inc	hl

	ld	d,0		; Load lower part of block of new map
	ld	e,c
	add	hl,de
	ld	(hl),a
	inc	hl
	ld	(hl),a
	dec	hl
	or	a
	sbc	hl,de

	ld	de,(temp_word)	; Check for end of line
	ex	de,hl
	sbc	hl,de
	ex	de,hl
	jr	nz,check_for_end
	ld	d,0		; End of line, so adjust to detect next
	ld	e,c
	add	hl,de
	inc	hl
	inc	hl
	
	push	hl		; Store address of end of line
	push	bc
	ld	b,0
	add	hl,bc
	inc	hl
	inc	hl
	ld	(temp_word),hl
	pop	bc
	pop	hl

check_for_end:
	pop	de
	inc	de
	djnz	enlarge_line
	ret

; ***** Enlarges gen_map4 to the size of the background, and copies it
;	onto the background
enlarge_to_background:
	ld	de,gen_map4	; Setup variables
	ld	hl,actual_map

	ld	b,32
another_line:
	push	bc
	ld	b,8
	ld	c,24
process_line:
	push	bc
	push	de
fill_line:
	ld	b,2
	ld	a,(de)
	cp	0
	jr	z,fill_block
	ld	a,$FF
fill_block:
	ld	(hl),a		; Fill first line of first block
	inc	hl
	djnz	fill_block
	inc	de
	dec	c
	jr	nz,fill_line

	pop	de
	pop	bc
	djnz	process_line

	push	hl
	ex	de,hl
	ld	de,24
	add	hl,de
	ex	de,hl
	pop	hl

	pop	bc
	djnz	another_line
	
	ret

; ***** Randomly adds/removes blocks to map at (HL) of length DE, and
;	width C bytes - modifies A
random_build_map:
	push	bc
	push	de
	push	hl

do_upwards_check:
	push	de
	push	hl
	ld	a,(hl)		; Get current block
	ld	b,0
	add	hl,bc
	ld	e,(hl)		; Get block below
	or	a
	sbc	hl,bc
	or	a
	sbc	hl,bc
	ld	d,(hl)		; Get block above
	pop	hl

	cp	0
	jr	z,cur_zero
cur_set:
	ld	a,d
	cp	0
	jr	z,cur_set_above_zero
cur_set_above_set:
	ld	a,e
	jr	z,cur_set_above_set_below_zero
cur_set_above_set_below_set:
	jr	done_block
cur_set_above_set_below_zero:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block
cur_set_above_zero:
	ld	a,e
	jr	z,cur_set_above_zero_below_zero
cur_set_above_zero_below_set:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block
cur_set_above_zero_below_zero:
	jr	done_block

cur_zero:
	ld	a,d
	cp	0
	jr	z,cur_zero_above_zero
cur_zero_above_set:
	ld	a,e
	jr	z,cur_zero_above_set_below_zero
cur_zero_above_set_below_set:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block
cur_zero_above_set_below_zero:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block
cur_zero_above_zero:
	ld	a,e
	jr	z,cur_zero_above_zero_below_zero
cur_zero_above_zero_below_set:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block
cur_zero_above_zero_below_zero:
	jr	done_block

done_block:
	inc	hl
	pop	de		; Check DE bytes
	dec	de
	xor	a
	cp	e
	jr	nz,do_upwards_check
	cp	d
	jr	nz,do_upwards_check

	pop	hl
	pop	de
	pop	bc
	ret

; ***** Randomly adds/removes blocks to map at (HL) of length DE,
;	only sideways not upwards - modifies A
random_build_map_side:
	push	bc
	push	de
	push	hl

do_side_check:
	push	de
	push	hl
	ld	a,(hl)		; Get current block
	dec	hl
	ld	e,(hl)		; Get block to left
	inc	hl
	inc	hl
	ld	d,(hl)		; Get block to right
	pop	hl

	cp	0
	jr	z,cur_zero2
cur_set2:
	ld	a,d
	cp	0
	jr	z,cur_set_right_zero
cur_set_right_set:
	ld	a,e
	jr	z,cur_set_right_set_left_zero
cur_set_right_set_left_set:
	jr	done_block2
cur_set_right_set_left_zero:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block2
cur_set_right_zero:
	ld	a,e
	jr	z,cur_set_right_zero_left_zero
cur_set_right_zero_left_set:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block2
cur_set_right_zero_left_zero:
	jr	done_block2

cur_zero2:
	ld	a,d
	cp	0
	jr	z,cur_zero_right_zero
cur_zero_right_set:
	ld	a,e
	jr	z,cur_zero_right_set_left_zero
cur_zero_right_set_left_set:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block2
cur_zero_right_set_left_zero:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block2
cur_zero_right_zero:
	ld	a,e
	jr	z,cur_zero_right_zero_left_zero
cur_zero_right_zero_left_set:
	call	get_rand_bit
	ld	(hl),a
	jr	done_block2
cur_zero_right_zero_left_zero:
	jr	done_block2

done_block2:
	inc	hl
	pop	de		; Check DE bytes
	dec	de
	xor	a
	cp	e
	jr	nz,do_side_check
	cp	d
	jr	nz,do_side_check

	pop	hl
	pop	de
	pop	bc
	ret

; ***** Sets the relevant variables to allow real-time moving of the
;	screen to the current worm
set_move_to_current_worm:
	push	hl
	push	de
	push	bc

	call	find_current_worm
	inc	hl
	inc	hl
	inc	hl
	ld	a,(hl)		; Get x position of worm
	and	%00000001
	ld	b,a
	inc	hl
	ld	a,(hl)
	srl	b		; Divide x-pos by 8 to get byte-pos
	rra
	srl	a
	srl	a
	ld	b,7		; Calculate x-byte pos of top left
	sub	b		; hand corner of screen to show worm
	cp	49
	jr	c,worm_screen_x_not_less_zero
	xor	a
worm_screen_x_not_less_zero:
	cp	33		; Check target x-pos does not exceed
	jr	c,store_new_x	; map boundary when displayed
	ld	a,32
store_new_x:
	ld	(target_x),a

	inc	hl
	ld	a,(hl)		; Get y-pos
	rrca			; Divide by 8 to get byte-pos
	rrca
	rrca
	and	%00011111
	ld	b,3		; Calculate y-byte pos of top left
	sub	b		; hand corner of screen to show worm
	cp	33
	jr	c,worm_screen_y_not_less_zero
	xor	a
worm_screen_y_not_less_zero:
	cp	25		; Check target y-pos does not exceed
	jr	c,store_new_y	; map boundary when displayed
	ld	a,24
store_new_y:
	ld	(target_y),a

	ld	a,(mode)
	set	m_finding_worm,a
	ld	(mode),a

	pop	bc
	pop	de
	pop	hl
	ret

; ***** Handle timing out of the turn timer
turn_timed_out:
	di
	ld	a,%00000010	; Only show message
	ld	(mode),a

	call	setup_message_buffer
	ld	hl,timed_out_message	; Copy message to graph backup buffer
	call	_vputs
	call	set_message_length
	call	redraw_map

wait_until_message_cleared:
	halt				; Switch to lower power mode until
	ld	a,(mode)		; message is cleared
	bit	m_showing_message,a
	jr	nz,wait_until_message_cleared

	call	set_next_worm		; Switch to next worm
	call	set_next_turn
	ret

; ***** Setup standard variables at the start of a turn
set_next_turn:
	di
	ld	a,%00001011
	ld	(mode),a
	ld	a,90
	ld	(worm_aim_direction),a
	ld	hl,MESSAGE_DELAY
	ld	(message_timer),hl
	ld	a,TURN_TIME
	ld	(turn_time_left),a
	xor	a
	ld	(x_map),a	; Clear/setup variables
	ld	(y_map),a
	ld	(x_dis),a
	ld	(cur_weapon_select),a
	ld	(slow_key_rep),a
	ld	h,0
	ld	l,h
	ld	(x_dis+1),hl
	ld	(y_dis),hl
	ld	(active_flash_on),a
	ld	(key_rep),hl
	call	write_active_worm_message
	call	set_move_to_current_worm
	ei
	ret

; ***** Set the appropriate variables to make it the next worm's turn
;	(and set the old worm's action to none)
set_next_worm:
	di
	call	find_current_worm
	inc	hl
	inc	hl
	inc	hl
	ld	a,(hl)
	and	%00011111
	ld	(hl),a

	ld	a,(cur_worm)
	ld	c,a
	inc	c
	ld	a,(total_num_worms)
	cp	c
	jr	z,set_first_worm
	ld	a,c
	ld	(cur_worm),a
	ret
set_first_worm:
	xor	a
	ld	(cur_worm),a
	ei
	ret

; ***** Controls firing
do_firing:
	ld	a,%01000000	; Only show weapon charging bar
	ld	(mode),a
	xor	a
	ld	(initial_vel),a

do_charge_weapon:
	push	hl
	call	q_get_key
	pop	hl
	cp	%00010000		; Check for release of Enter key
	jr	nz,actually_fire_weapon
	ld	a,(initial_vel)
	cp	128
	jr	z,actually_fire_weapon
	inc	a
	ld	(initial_vel),a
	call	redraw_map
	halt
	jr	do_charge_weapon

; Calculate the initial coords (displacement relative to top left of map) of
; projectile, as well as velocity in x and y directions.  Initially, x_vel
; and y_vel hold the screen-relative displacements of the projectile, while
; x_dis and y_dis hold the displacement relative to the worm (1 byte for all)
actually_fire_weapon:
	di
	ld	a,%00000100	; Only track projectile
	ld	(mode),a

	ld	a,(x_dis)	; Store relative displacements in fractional
	ld	(x_dis+2),a	; part of displacement variables
	ld	a,(y_dis)
	ld	(y_dis+1),a

	ld	a,(x_dis+2)	; Convert relative displacement if negative
	bit	7,a
	jr	z,no_conv_dis_x
	ld	b,a
	xor	a
	sub	b
	set	7,a
	ld	(x_dis+2),a
no_conv_dis_x:

	ld	a,(y_dis+1)
	bit	7,a
	jr	z,no_conv_dis_y
	ld	b,a
	xor	a
	sub	b
	set	7,a
	ld	(y_dis+1),a
no_conv_dis_y:

	ld	a,(x_map)	; Alter x_dis so that it holds the pixel
	ld	h,0		; displacement of the projectile relative to
	ld	l,a		; the top left of the map - Note that when
	ld	a,(x_vel)	; calculating aim sprite coords, x_dis and
	ld	d,h		; y_dis are relative, i.e. they are added to
	ld	e,a		; the worm's coords to get the sprite coords.
	add	hl,hl		; x_vel and y_vel holds absolute coords,
	add	hl,hl		; though
	add	hl,hl
	add	hl,de
	ld	a,h
	ld	(x_dis),a
	ld	a,l
	ld	(x_dis+1),a

	ld	a,(y_map)	; Alter y_dis in the same way as x_dis
	rla
	rla
	rla
	and	%11111000
	ld	b,a
	ld	a,(y_vel)
	add	a,b
	ld	(y_dis),a

	ld	(wind),a	; No wind effect

	ld	a,(initial_vel)	; Reduce initial velocity
	srl	a
	srl	a
	ld	(initial_vel),a
	
	ld	a,(initial_vel)	; Calculate x-velocity by multiplying
	ld	h,a		; aim sprite x-displacement by force
	ld	l,0
	ld	a,(x_dis+2)
	res	7,a
	ld	e,a
	ld	d,l
	call	mul8

	ld	d,0
	ld	a,(x_dis+2)	; Check for negative displacement, and alter
	bit	7,a		; if needed
	jr	z,not_neg_x_dis
	
	ld	d,%10000000
not_neg_x_dis:
	ld	a,d		; Store sign bit in D
	ld	(x_vel),a
	ld	a,h		; Store calculated x velocity
	ld	(x_vel+1),a
	ld	a,l
	ld	(x_vel+2),a

	ld	a,(initial_vel)	; Calculate y-velocity by multiplying
	ld	h,a		; aim sprite y-displacement by force
	ld	l,0
	ld	a,(y_dis+1)
	res	7,a
	ld	e,a
	ld	d,l
	call	mul8

	ld	d,0
	ld	a,(y_dis+1)	; Check for negative displacement, and alter
	bit	7,a		; if needed
	jr	z,not_neg_y_dis
	
	ld	d,%10000000
not_neg_y_dis:
	ld	a,h
	or	d		; Add on sign bit in D
	ld	(y_vel),a
	ld	a,l
	ld	(y_vel+1),a

	xor	a		; Clear fractional parts of displacements
	ld	(x_dis+2),a
	ld	(y_dis+1),a

	ld	a,a_stopped	; Get rid of aim cursor
	call	set_current_worm_action

update_projectile:
	call	redraw_map	; Track projectile until it hits something
	halt
	ld	a,(mode)
	bit	m_firing,a
	jr	nz,update_projectile

	call	set_next_worm		; Switch to next worm
	call	set_next_turn
	ret

; ***** Tracks a projectile that has been fired.  Note that x_vel and y_vel
;	are signed, but not by 2's complement.  They are negative if their
;	MS bit is set, but the rest of the number is zero based, e.g. $FF
;	is not -1, but -127.
track_projectile:

; Update x-displacement
	ld	a,(x_dis)
	bit	7,a
	jr	z,calc_pos_x_dis

	res	7,a		; Calculate by subtracting, rather than
	ld	a,(x_dis+2)	; adding
	ld	b,a
	ld	a,(x_vel+2)
	sub	b
	ld	(x_dis+2),a
	ld	a,(x_dis+1)
	ld	b,a
	ld	a,(x_vel+1)
	sbc	a,b
	ld	(x_dis+1),a
	ld	a,(x_dis)
	ld	b,a	
	ld	a,(x_vel)
	sbc	a,b
	ld	(x_dis),a
	jr	fin_calc_x_dis

calc_pos_x_dis:
	ld	a,(x_dis+2)	; Add LSB of velocity to LSB of displacement
	ld	b,a
	ld	a,(x_vel+2)
	add	a,b
	ld	(x_dis+2),a
	ld	a,(x_dis+1)	; Add mid-byte of velocity to mid-byte of
	ld	b,a		; displacement
	ld	a,(x_vel+1)
	adc	a,b
	ld	(x_dis+1),a
	ld	a,(x_dis)	; Add MSB of velocity to MSB of displacement
	ld	b,a	
	ld	a,(x_vel)
	adc	a,b
	ld	(x_dis),a

fin_calc_x_dis:

; Update y-displacement
	ld	a,(y_dis)
	bit	7,a
	jr	z,calc_pos_y_dis

	res	7,a
	ld	a,(y_dis+1)	; Calculate by subtracting, rather than
	ld	b,a		; adding
	ld	a,(y_vel+1)
	sub	b
	ld	(y_dis+1),a
	ld	a,(y_dis)
	ld	b,a	
	ld	a,(y_vel)
	sbc	a,b
	ld	(y_dis),a
	jr	fin_calc_y_dis

calc_pos_y_dis:
	ld	a,(y_dis+1)	; Add LSB of velocity to LSB of displacement
	ld	b,a
	ld	a,(y_vel+1)
	add	a,b
	ld	(y_dis+1),a
	ld	a,(y_dis)	; Add MSB of velocity to MSB of displacement
	ld	b,a	
	ld	a,(y_vel)
	adc	a,b
	ld	(y_dis),a

fin_calc_y_dis:

; Calculate screen x-pos
	ld	a,(x_map)	; Calculate position of projectile on screen
	call	mul_a_by_8
	ld	a,(x_dis)
	ld	d,a
	ld	a,(x_dis+1)
	ld	e,a
	ex	de,hl
	or	a
	sbc	hl,de
	push	hl

; Calculate screen y-pos
	ld	a,(y_map)
	call	mul_a_by_8
	ld	a,(y_dis)
	ld	d,0
	ld	e,a
	ex	de,hl
	or	a
	sbc	hl,de

	ld	d,l	; Show projectile sprite
	pop	hl
	ld	e,l
	ld	hl,missile_sprite
	call	PutSprite

; Check for collision
	

	ret

; ***** Draw the weapon charging bar (force in initial_vel)
show_weapon_charging_bar:
	.db	$11,$00		; ld DE,cur_video_page_address
do_firing_video_addr:
	.db	$FC
	ld	hl,656		; Draw clear bar
	add	hl,de
	push	hl
	ld	bc,160-1
	ld	d,h
	ld	e,l
	inc	de
	xor	a
	ld	(hl),a
	ldir

	pop	hl
	ld	a,(initial_vel)	; Max. velocity=128
	cp	0
	ret	z		; Just show empty bar if vel=0
	cp	129
	ret	nc
	push	hl
	ld	de,16		; Get address of centre part of bar
	add	hl,de

	rrca			; Divide velocity by 8 to get number of
	rrca			; whole bytes occupied by velocity bar
	rrca
	and	%00011111
	cp	0
	jr	z,calc_bar_end
	ld	b,a		; Calculate amount to add to video write
	ld	a,16		; address each time to move to the next line
	sub	b
	ld	e,a
	ld	d,0
	ld	c,8
	ld	a,$ff
show_bar_line:
	push	bc
show_bar_byte:
	ld	(hl),a
	inc	hl
	djnz	show_bar_byte
	add	hl,de
	pop	bc
	dec	c
	jr	nz,show_bar_line

calc_bar_end:
	pop	hl
	ret

; ***** Sets the current worm's action (mask value for action in B)
set_current_worm_action:
	push	hl
	push	bc
	call	find_current_worm
	pop	bc
	inc	hl
	inc	hl
	inc	hl
	ld	a,(hl)		; Get worm's action, and mask off it's
	and	%00011111	; old action
	or	b		; Set new action
	ld	(hl),a

	pop	hl
	ret

; ***** Returns the address of the data for the currently active worm
;	in HL
find_current_worm:
	ld	a,(cur_worm)	; Find address of current worm data
	ld	l,a
	ld	h,0
	add	hl,hl
	add	hl,hl
	add	hl,hl
	ld	bc,worm_data
	add	hl,bc
	ret

; ***** Returns values in x_dis and y_dis that when added to the current
;	worm sprite coords give the coords of the aim_sprite
get_aim_sprite_coords:
	call	convert_to_real_angle

	push	hl
	call	_setxxxxOP2
	call	_OP2TOOP1
	call	_COS		; Find cosine of angle, and multiply by
	ld	a,AIM_DISTANCE	; AIM_DISTANCE to find X-coord change
	call	_setxxOP2
	call	_FPMULT
	call	conv_op1_to_a
	ld	(x_dis),a

	pop	hl
	call	_setxxxxOP2
	call	_OP2TOOP1
	call	_SIN		; Find sine of angle, and multiply by
	ld	a,AIM_DISTANCE	; AIM_DISTANCE to find Y-coord change
	call	_setxxOP2
	call	_FPMULT
	call	conv_op1_to_a
	ld	(y_dis),a
	ret

; ***** Converts OP1 to a signed value in A - modifies A and DE
conv_op1_to_a:
	call	_CONVOP1	; X-adjustment value in DE
	ld	a,(_OP1)
	bit	7,a
	jr	nz,neg_op_value
	ld	a,e
	ret
neg_op_value:
	ld	a,e
	neg
	ret

; ***** Converts worm_aim_direction along with the direction of the current
;	worm
convert_to_real_angle:
	call	find_current_worm
	inc	hl
	inc	hl
	ld	a,(hl)
	and	%11110000	; Extract current worm's direction
	cp	d_left
	ld	a,(worm_aim_direction)	; Put relative worm aim angle into BC
	ld	c,a
	ld	b,0
	jr	nz,calc_angle_right
	ld	hl,270		; Find real aiming angle (left only)
	or	a
	sbc	hl,bc
	ret
calc_angle_right:
	cp	90		; Use alternate method for angle greater than
	jr	nc,calc_angle_greater_90	; 90 degrees
	ld	hl,270
	add	hl,bc		; Just add 270 degrees to find real angle
	ret
calc_angle_greater_90:
	ld	h,b		; just subtract 90 degress to find real angle
	ld	l,c
	ld	bc,90
	or	a
	sbc	hl,bc
	ret

; ***** Quickly gets a key through direct reading of the key port, and
;	returns a scan-code in A (upper-nibble = row, lower-nibble
;	= column) - All registers modified
q_get_key:
	ld	c,%11111110
	ld	a,c
col_scan_loop:
	di			; Checks each row until a pressed
	out	(1),a		; button is found, or until all
	nop			; rows are checked
	nop
	in	a,(1)
	ei
	cp	$ff
	jr	nz,store_col
	rlc	c
	ld	a,c
	cp	%01111111
	jr	nz,col_scan_loop
	ld	a,$FF
	ret

store_col:
	ld	b,6		; Row goes into A, column into E
	ld	d,%01000000
	ld	e,a
	ld	a,c
store_col_loop:
	and	d
	ld	a,c
	jr	z,row_store_loop
	rrc	d
	dec	b
	jr	store_col_loop

row_store_loop:
	ld	a,e		; Row number into upper-nibble
	ld	c,a		; of E, column into A
	ld	e,b
	sla	e
	sla	e
	sla	e
	sla	e
	ld	b,7
	ld	d,%10000000
store_row_loop:
	and	d
	ld	a,c
	jr	z,ret_scan_code
	rrc	d
	dec	b
	jr	store_row_loop

ret_scan_code:
	ld	a,e		; Column number into lower-nibble of A
	add	a,b
	ret

; ***** Loads a random value into A
get_rand:
	push	hl
	push	de
	ld	hl,(rand_seed)
	ld	a,(hl)
	ld	b,a
	ld	a,r
	ld	e,a
	ld	d,a
	add	hl,de
	ld	(rand_seed),hl
	ld	a,b
	pop	de
	pop	hl
	ret

; ***** Loads either 0 or 1 into A, randomly
get_rand_bit:
	call	get_rand
	cp	$E8
	jr	c,less80
	ld	a,1
	ret
less80:
	xor	a
	ret

; ***** Multiplies contents of A by 8, and stores in HL
mul_a_by_8:
	ld	l,a
	ld	h,0
	add	hl,hl
	add	hl,hl
	add	hl,hl
	ret

; ***** Display number in HL using variable width font
V_D_HL_DECI:
         push     bc
         ld       de,disp_data+4
         ld       b,5
ldhld:   call     UNPACK_HL
         add      a,48
         ld       (de),a
         dec      de
         djnz     ldhld
         ld       hl,disp_data
         ld       b,4
lis:     ld       a,(hl)
         cp       48
         jr       nz,dis
         ld       (hl),32
         inc      hl
         djnz     lis
dis:     ld       hl,disp_data
         call     _vputs
         pop	  bc
         ret

; ***** Switches display pages - also updates display routines code so
;       that they write to the correct page (the one that isn't being
;	displayed
switch_display:
	ld	a,(cur_page)	; Get current page, and compare to
	cp	disp_page_1	; value of first
	jr	z,switch_to_2
switch_to_1:
	ld	a,disp_page_1
	ld	(cur_page),a
	out	(0),a
	ld	a,disp_page_2
	jr	update_disp_routines
switch_to_2:
	ld	a,disp_page_2
	ld	(cur_page),a
	out	(0),a
	ld	a,disp_page_1
update_disp_routines:
	add	a,$C0
	ld	(find_pixel_write_addr),a	; Update FindPixel
	ld	(redraw_map_write_addr),a	; Update redraw_map
	ld	(check_valid_ram_addr),a
	ld	(show_message_video_addr),a
	ld	(weapon_select_video_addr),a
	ld	(do_firing_video_addr),a
	ret

; ***** Changes display routines to write to the currently visible
;	viewport
write_to_current_display:
	ld	a,(cur_page)
	add	a,$C0
	ld	(find_pixel_write_addr),a  ; Update FindPixel
	ld	(redraw_map_write_addr),a  ; Update redraw_map
	ld	(check_valid_ram_addr),a
	ld	(show_message_video_addr),a
	ld	(weapon_select_video_addr),a
	ld	(do_firing_video_addr),a
	ret

; ***** Clear current page
clear_current_page:
	ld	a,(cur_page)
	ld	h,a
	xor	a
	ld	l,a
	ld	b,a
	ld	c,4
do_clear_page:
	ld	(hl),a
	djnz	do_clear_page
	dec	c
	jr	nz,do_clear_page

; ***** Copies other display page to visible display page
copy_to_display:
	ld	a,(cur_page)
	add	a,$C0
	ld	d,a
	ld	e,0
	and	disp_page_1
	jr	z,copy_to_1
copy_to_2:
	ld	h,disp_page_1+$C0
	ld	l,e
	jr	do_copy_display
copy_to_1:
	ld	h,disp_page_2+$C0
	ld	l,e
do_copy_display:
	ld	bc,1024
	ldir
	ret

; ***** Fills B bytes of memory starting at HL (max. 256 bytes) with A
fill_8_bit:
	ld	(hl),a
	inc	hl
	djnz	fill_8_bit
	ret

; ***** Multiplies AC by DE, returns result in HL
mul16:
	ld	b,16
	ld	hl,0
do_mult16:
	srl	c
	rra
	jr	nc,no_add16
	add	hl,de
no_add16:
	ex	de,hl
	add	hl,hl
	ex	de,hl
	djnz	do_mult16
	ret

; ***** Multiplies H by E, returns result in HL
mul8:
	ld	b,8
do_mult8:
	add	hl,hl
	jr	nc,no_add8
	add	hl,de
no_add8:
	djnz	do_mult8
	ret

;--------------------------------------------------------------------
; The Eble-Yopp-Yopp-Eble-Eble-Eble-Yopp Fast FindPixel Routine :)
; 36 bytes / 121 t-states not counting ret or possible push/pop of BC
;--------------------------------------------------------------------
; Input: D = y; E = x; Output: HL= address of byte in video memory
; A = bitmask (bit corresponding to pixel is set)
; C is modified
;
; +-----------+
; |(0,0)      | <- Screen layout
; |           |
; | (127,63)  |
; +-----------+
;
;--------------------------------------------------------------------
FindPixel:
	ld hl,FP_Bits
	ld a,e
	and $07 ; a = bit offset
	add a,l
	ld l,a
	adc a,h
	sub l
	ld h,a
	ld c,(hl) ; c = bitmask for (hl)
; 48 t-states up to this point
	ld hl,FP_RLD
	ld (hl),d
	ld a,e ; a = x/8 (byte offset within row)
	rrca
	rrca
	rrca
	rld
	.db $F6 ; or $FC
find_pixel_write_addr:
	.db $FC
	ld l,(hl)
	ld h,a ; hl -> byte in vid mem
	ld a,c ; now a = bitmask for (hl)
; 121 t-states up to this point
	ret
FP_RLD: .db $00
FP_Bits: .db $80,$40,$20,$10,$08,$04,$02,$01

; ***** PutWorm routine - Draws the numbered worm sprite (number in A)
;	to the screen at D=y, E=x - All registers modified
PutWorm:

; First, get worm data address in HL.  Each worm is 8x12 pixels
; in size + 1 size byte, and so occupies 13 bytes.  Just multiply A by
; 8, then add on original value 5 times
	ld	b,0
	ld	c,a
	ld	h,b
	ld	l,a
	add	hl,hl
	add	hl,hl
	add	hl,hl
	add	hl,bc
	add	hl,bc
	add	hl,bc
	add	hl,bc
	add	hl,bc
	ld	bc,worm_sprites
	add	hl,bc		; HL now points to worm sprite
	jp	PutSprite

; ***** PutSprite routine - Draws sprite at (HL) to screen at D=y,
;       E=x - All registers modified
;
;	Note - Wraps sprites down to next line
;
PutSprite:

; Find the lower 3 bits, and if they are all zero draw sprite at the
; start of a byte of video memory.  Otherwise, calculate overlap from
; one byte to another
	ld	a,(hl)	; Get size
	ld	b,a
	inc	hl
put_byte:
	push	bc
	push	hl	; Store sprite address
	ld	a,e
	and	%00000111	; Find amount to shift sprite data byte
	ld	b,a		; left by (8-amount to shift right)
	ld	a,8
	sub	b
	ld	b,a
	ld	l,(hl)
	ld	h,0
shift_sprite:
	add	hl,hl
	djnz	shift_sprite

draw_sprite:
	push	hl
	call	FindPixel	; Find location in video RAM to write
				; pixels

	bit	7,d		; Check for negative y-pos
	jr	z,no_neg_y
	ld	bc,$400
	or	a
	sbc	hl,bc

no_neg_y:
	pop	bc
	push	de		; Check video RAM address is valid
	push	hl
	.db	$11,$00		; ld DE,cur_video_page_address
check_valid_ram_addr:
	.db	$FC
	or	a		; Check for too low address
	sbc	hl,de
	jr	c,invalid_address

	ld	de,$400		; Check for too high address
	or	a
	sbc	hl,de
	pop	hl
	pop	de
	jr	nc,no_show_all
	jr	check_x_coord

invalid_address:
	pop	hl
	pop	de
	jr	no_show_all

check_x_coord:
	ld	a,120			; Check for x-coord out of bounds
	cp	e			; (or partly out of bounds
	jr	nc,cont_draw_sprite
	ld	a,-8
	cp	e
	jr	c,draw_right_byte_only
	ld	a,127
	cp	e
	jr	c,no_show_all

draw_left_byte_only:
	ld	a,(hl)
	xor	b
	ld	(hl),a
	inc	hl
	jr	no_show_all

draw_right_byte_only:
	inc	hl
	ld	a,(hl)
	xor	c
	ld	(hl),a
	jr	no_show_all

cont_draw_sprite:
	ld	a,(hl)
	xor	b
	ld	(hl),a
	inc	hl
	ld	a,(hl)
	xor	c
	ld	(hl),a
no_show_all:
	
	pop	hl	; Update sprite data byte address
	inc	hl
	pop	bc
	inc	d
	ld	a,64
	cp	d
	ret	z
	djnz	put_byte
	ret

; ***** Writes a value to the graph backup buffer (value in A), and
;	then to the value_sprite sprite storage space
writeNumericValue:
	push	af
	ld	hl,value_buffer
	xor	a
	ld	b,128
	call	fill_8_bit
	pop	af

	ld	l,a		; Write value to graph backup
	ld	h,0		; buffer
	xor	a
	ld	(_penCol),a
	ld	a,57
	ld	(_penRow),a
	call	V_D_HL_DECI

	ld	b,6		; Write value from buffer to a
	ld	hl,value_buffer	; sprite holder
	ld	de,value_sprite1+1
copy_value_line:
	ld	a,(hl)
	ld	(de),a
	push	bc
	ld	bc,16
	add	hl,bc
	pop	bc
	inc	de
	djnz	copy_value_line

	ld	hl,value_sprite1	; Set size of sprite
	ld	a,6
	ld	(hl),a

	ld	b,6
	ld	hl,value_buffer+1	; Write remainder of value
	ld	de,value_sprite2+1	; image to second sprite holder
copy_value_line2:
	ld	a,(hl)
	ld	(de),a
	push	bc
	ld	bc,16
	add	hl,bc
	pop	bc
	inc	de
	djnz	copy_value_line2

	ld	hl,value_sprite2
	ld	a,6
	ld	(hl),a
	ret

; ***** Copies the weapon select menu bar to the current video page
show_weapon_select:
	.db	$11,$00		; ld DE,cur_video_page_address
weapon_select_video_addr:
	.db	$FC
	ld	hl,656
	add	hl,de
	push	hl
	ex	de,hl
	ld	hl,menu_bar_sprite	; Copy menu bar to video page
	ld	bc,160			; (16 X 8 byte sprite)
	ldir
	pop	hl

	ld	d,0			; Invert selected menu item
	ld	a,(cur_weapon_select)
	ld	e,a
	add	hl,de
	ld	b,10
	ld	c,$FF
	ld	de,16
invert_menu_item:
	ld	a,(hl)
	xor	c
	ld	(hl),a
	add	hl,de
	djnz	invert_menu_item

	ld	a,(mode)		; Disable any other messages, and
	res	m_showing_message,a	; enable a fixed one for the weapon
	set	m_fixed_message,a	; names
	ld	(mode),a

	ld	a,(cur_weapon_select)
	ld	b,a
	ld	hl,weapon_names
	cp	0
	jr	z,done_finding_weapon_name
find_weapon_name:
	ld	a,(hl)			; Find the name of the weapon by
	inc	hl			; scanning through the list of names
	cp	0
	jr	nz,find_weapon_name
	djnz	find_weapon_name
done_finding_weapon_name:

	push	hl
	call	setup_message_buffer
	pop	hl
	call	_vputs
	call	set_message_length
	ret

; ***** Copies the message in the graph backup buffer to the bottom of
;	the current video page
show_message:
	.db	$11,$00		; ld DE,cur_video_page_address
show_message_video_addr:
	.db	$FC
	ld	hl,$3A0		; Get address of start of last 6 lines
	add	hl,de		; of video RAM
	ld	d,0
	ld	a,(message_x)	; Adjust address for x-pos of message
	ld	e,a
	add	hl,de
	ex	de,hl
	ld	hl,message_sprite
	ld	b,96
do_xor_message:
	ld	c,(hl)		; XOR message with display
	ld	a,(de)
	xor	c
	ld	(de),a
	inc	hl
	inc	de
	djnz	do_xor_message
	ret

; ***** Writes a message to the sprite holder indicating the current
;	active worm name and team number
write_active_worm_message:
	call	setup_message_buffer
	ld	hl,new_active_worm_message1	; Write message to
	call	_vputs			; graph backup buffer (1st part)

	ld	a,(cur_team)		; Write team number to sprite
	inc	a
	add	a,$30
	ld	(disp_data+4),a
	ld	hl,disp_data+4
	call	_vputs

	ld	hl,new_active_worm_message2	; Write 2nd part of
	call	_vputs				; message

	ld	hl,worm_name_table	; Get name of current worm
	ld	d,0			; and write to sprite
	ld	a,(cur_worm)
	ld	e,a
	sla	e
	add	hl,de
	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	ex	de,hl
	call	_vputs

	ld	hl,exclamation_string		; Write exclamation
	call	_vputs				; mark
	call	set_message_length
	ret

; ***** Sets up the message buffer for writing to - modifies A and HL
setup_message_buffer:
	ld	hl,message_sprite	; Clear sprite
	xor	a
	ld	b,96
	call	fill_8_bit

	xor	a
	ld	(_penCol),a
	ld	a,50
	ld	(_penRow),a
	ret

; ***** Calculates the x-position that the message in the message buffer
;	must be given for it to be centred on screen - modifies A and B
set_message_length:
	ld	a,(_penCol)		; Get message x-pos
	srl	a
	ld	b,a
	ld	a,64
	sub	b
	srl	a
	srl	a
	srl	a
	ld	(message_x),a
	ret

; ***** Interrupt handling routine - outside of main interrupt handler
;	so there is no limit to it's size
handle_interrupt:

; Check whether active worm indicator needs to be flashed
	ld	a,(mode)
	bit	m_worm_playing,a
	jr	z,check_for_remove_message

	ld	hl,(active_flash_timer)
	ld	de,0
	or	a
	sbc	hl,de
	jr	nz,update_flash_timer
	ld	hl,FLASH_ACTIVE_DELAY
	ld	(active_flash_timer),hl
	ld	a,(active_flash_on)	; Complement "flash on" value
	rra
	ccf
	rla
	ld	(active_flash_on),a
	ld	a,1			; Specify redraw needed
	ld	(int_redraw_check),a
update_flash_timer:
	ld	hl,(active_flash_timer)
	dec	hl
	ld	(active_flash_timer),hl

; Check whether displayed message needs to be removed
check_for_remove_message:
	ld	a,(mode)
	bit	m_showing_message,a
	jr	z,check_for_timer_update

	ld	hl,(message_timer)
	ld	de,0
	or	a
	sbc	hl,de
	jr	nz,update_message_timer
	res	m_showing_message,a
	ld	(mode),a
	ld	hl,MESSAGE_DELAY
	ld	(message_timer),hl
	ld	a,1			; Specify redraw needed
	ld	(int_redraw_check),a
update_message_timer:
	ld	hl,(message_timer)
	dec	hl
	ld	(message_timer),hl

; Check whether timer needs to be updated
check_for_timer_update:
	ld	a,(mode)
	bit	m_worm_playing,a
	jr	z,check_for_redraw

	ld	hl,(timer_count)
	ld	de,0
	or	a
	sbc	hl,de
	jr	nz,update_timer_count
	ld	hl,TIMER_DELAY
	ld	(timer_count),hl
	ld	a,(turn_time_left)
	cp	0
	jr	z,timer_at_zero_just_redraw
	dec	a
	ld	(turn_time_left),a
timer_at_zero_just_redraw:
	ld	a,1			; Specify redraw needed
	ld	(int_redraw_check),a
update_timer_count:
	ld	hl,(timer_count)
	dec	hl
	ld	(timer_count),hl

check_for_redraw:
	ret

; ***** Interrupt routine
IntRoutine:
	di			; **** MUST SWITCH OFF INTERRUPTS ****
	pop	hl		; Fast interrupt!
	exx
	ex	af,af'

	push	af
	push	bc
	push	de
	push	hl

	in	a,(5)		; **** MUST SAVE ROM PAGE ****
	push	af
	ld	a,$0d
	out	(5),a

	call	handle_interrupt

; Check whether changes to the display by this interrupt require it to
; be redrawn
	ld	a,(int_redraw_check)
	cp	0
	jr	z,check_for_display_update
	xor	a
	ld	(int_redraw_check),a
	in	a,(6)
	push	af
	ld	a,$41		; Set paged RAM to map page (MUST
	out	(6),a		; DO THIS, OR WILL GET RANDOM CRASHES)
	call	redraw_map
	di
	pop	af
	out	(6),a		; MUST RESTORE RAM PAGE

; Check whether display needs to be updated
check_for_display_update:
	ld	a,(interrupt_byte)
	bit	0,a
	jr	z,end_int_routine
	res	0,a	
	ld	(interrupt_byte),a

	in      a,(3)		; If VBL interrupt, then switch
        bit     1,a		; displays
        jr      z,end_int_routine

	call	switch_display	; Update display

end_int_routine:
 	in a,(3)	; Stop return to system interrupt routine
	rra
 	ld  a,9
 	adc a,0
 	out (3),a
 	ld a,$0B
	out (3),a

	pop	af
	out	(5),a

	pop	hl
	pop	de
	pop	bc
	pop	af

	ei
        reti
InterruptMemEnd:

; ***** String data
title_bar:
	.db	"Worms! V1.0 by David Hart",0
options_list:
	.db	"START GAME",0
	.db	"Number of teams:",0
	.db	"Worms per team:",0
disp_data:
	.db	"12345",0

; ***** Messages
timed_out_message:
	.db	"Too slow!",0
new_active_worm_message1:
	.db	"Team ",0
new_active_worm_message2:
	.db	" - Get 'em "
	.db	0
exclamation_string:
	.db	"!",0

; ***** Worm names (16)
actual_worm_names:
	.db	"Benjy",0
	.db	"Hardnut",0
	.db	"Whacker",0
	.db	"Smasher",0
	.db	"Mincer",0
	.db	"Fishman",0
	.db	"Macho",0
	.db	"Trosser",0
	.db	"Mallet",0
	.db	"Belcher",0
	.db	"Belter",0
	.db	"Zany",0
	.db	"Knuckles",0
	.db	"Crasher",0
	.db	"Mickey",0
	.db	"Flergywergyjergynergy",0

; ***** Weapon names (5)
weapon_names:
	.db	"Bazooka",0
	.db	"Grenade",0
	.db	"Dynamite",0
	.db	"Bash",0
	.db	"Kamikaze",0

; ***** Worm sprites
worm_sprites:

; 1 - Still worm
	.db	12
	.db	%01110000
	.db	%10001000
	.db	%10100100
	.db	%10000100
	.db	%01001000
	.db	%00101000
	.db	%00101001
	.db	%00101011
	.db	%01010101
	.db	%10100101
	.db	%10111010
	.db	%01111100

; ***** Other sprites

; Aiming pointer (cross-hair)
aim_sprite:
	.db	8
	.db	%00000000
	.db	%01100110
	.db	%01000010
	.db	%00011000
	.db	%00011000
	.db	%01000010
	.db	%01100110
	.db	%00000000

; Active worm indicator
active_sprite:
	.db	4
	.db	%11111111
	.db	%01111110
	.db	%00111100
	.db	%00011000

; Bazooka missile
missile_sprite:
	.db	8
	.db	%00000000
	.db	%00111100
	.db	%01111110
	.db	%11111111
	.db	%11111111
	.db	%01111110
	.db	%00111100
	.db	%00000000

; Menu bar icons
menu_bar_sprite:
  .db %00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000
  .db %00011000,%00111000,%00110000,%00000000,%01001010,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000
  .db %00111100,%00001000,%00001000,%00111100,%00101100,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000
  .db %00111100,%00001000,%00001000,%01000010,%01111000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000
  .db %01111110,%00011100,%00011100,%01000110,%01111110,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000
  .db %01111110,%00111110,%00011100,%00111110,%00111100,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000
  .db %01111110,%00111110,%00011100,%00000010,%01101110,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000
  .db %01111110,%00111110,%00011100,%00000100,%00101100,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000
  .db %01111110,%00011100,%00011100,%00000110,%01010010,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000
  .db %00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000,%00000000

	.end
