;
; If you make any changes to the source, please tell me what and why.
; And you are NOT allowed to distribute a modified source, nor the
; compiled version of it. Any changes should be made for personal use only.
;
; //Jimmy Mrdell <yarin@acc.umu.se>
;

#include sqrxz.h
#include ti86asm.inc
#include ti86abs.inc

XScr		   = $C0F9   ; X scroller
XBlock		   = $C0FC   ; XScr SHR 3 (1 block = 8x8 pixels)
LevelSize	   = $C0FD   ; Length of level (x blocks)
x		   = $C0FF   ; X location of Sqrxz
y		   = $C101   ; Y location of Sqrxz
flags		   = $C102   ; 1 - HScroll, 2 - Sqrxz has moved
			     ; 3 - Died this frame, 4 - Was under portcullis
			     ; 5 - Update enemies
jump		   = $C103   ; Where in the jump arc (backwards)
fall		   = $C104   ; Same as above, except when falling (and not bw)
sqrxzcnt	   = $C105   ; Sqrxz counter
sqrxzdir	   = $C106   ; Sqrxz direction (0 = Right, 1 = Left)
jump_button	   = $C107   ; The jump button status
dead		   = $C108   ; 0 = alive, >0 = dead
counter 	   = $C109   ; Main counter
animcnt 	   = $C10A   ; Animation counter, counts 0-2
hmovelast	   = $C10B   ; 1 = Update Sqrxz
deadcnt 	   = $C10C   ; Death pause counter - starts after death
timer		   = $C10D   ; Timer - increased every interrupt call (200 hz)
tmp_x		   = $C10E   ; Temporary storage for Sqrxz X
tmp_y		   = $C110   ; Temporary storage for Sqrxz Y
start_y 	   = $C111   ; Sqrxz start Y
start_x 	   = $C112   ; Sqrxz start X (0=start of level)
practice	   = $C113   ; 0=normal play, 1=practice mode
enemytable	   = $C120   ; Enemy table [0..7] type,dir,x(w),y,fall,spec

world		   = $C160   ; Address to selected world
worldB		   = $C162   ; RAM block of selected world
lvlptrs 	   = $C163   ; Pointer to level pointers, alt level info
worldT		   = $C165   ; 0=uncompressed levels, 1=compressed levels
noLevels	   = $C171   ; Number of levels on the selected world
lvl		   = $C172   ; Current level
lives		   = $C173   ; Lives left
timeleft	   = $C174   ; Time left
sausages	   = $C176   ; Sausages taken
gaming		   = $C177   ; 0=not playing, 1=playing
waterbar	   = $C178   ; The water bar (0-479)

clipmask	   = $C182   ; Temporary byte used in ASCR
rows2put	   = $C183   ; Temporary byte used in ASCR
bitmask 	   = $C184   ; Temporary byte used in ASCR

SqrxzBG 	   = $CFAB   ; Old Sqrxz X,Y and background (10 byte)
LastSprite	   = $CFB5   ; Pointer to the last enemy sprite put
BGETable	   = $CFB7   ; Table of old enemy data: X,Y and background
animptr 	   = $D007   ; Next change in the table (0-7)
AnimTable	   = $D008   ; Anim table. 8 entries, 4 byte each (x,y,addr)
portc		   = $D028   ; A portcullis that will go down (x,y,addr)

noWorlds	   = $CFAB   ; Number of worlds found on calc
top		   = $CFAC   ; Top of the world list
noItems 	   = $CFAD   ; Number of worlds in the current list
curItem 	   = $CFAE   ; Current world choice
curOpt		   = $CFAF   ; Current option
worlds		   = $CFB0   ; Pointers to different worlds

tmpWorld	   = $9000   ; Temporary storage for world to play
LevelPtr	   = $B000

.org _asm_exec_ram

 call _clrLCD
 call _flushallmenus
 call _runindicoff     ; Turn busy indicator off
 res appTextSave,(iy+appflags)

 ld hl,($4065)	       ; ROM patch, sort of.
 ld (intgetkey+1),hl   ; call $4064 doesn't work in an interrupt handler (?)

 ld hl,$8E00
 ld (hl),$8F
 ld bc,256
 call FillChar	       ; Create a 257 byte vector table
 ld hl,IntCounter      ; Make HL point to the interrupt handler
 ld de,$8F8F	       ; And DE where to store the handler
 ld bc,IntEnd-IntCounter
 ldir
 ld a,$8E
 ld i,a

 ld hl,TitlePic        ; HL -> titlepicture compressed data
 ld ix,_plotSScreen    ; IX -> graph memory (temporary storage)
 ld de,$FCF0	       ; DE -> where pic should be extracted
 call HuffExtr	       ; And call the routine to extract the pic
 ld hl,$2A22
 ld (_penCol),hl
 ld hl,Coder	       ; Show coder
 call _vputs
 ld de,$390F
 ld (_penCol),de
 call _vputs
TitleWait:
 call _getky
 or a
 jr z,TitleWait        ; And wait until a key is pressed
 xor a		       ; Clear some main variables
 ld (lvl),a	       ; Start at the first level (0, although displayed as 1)
 ld (sausages),a       ; Number of sausages eaten
 ld (gaming),a	       ; A flag telling the game hasn't started yet
 ld (start_x),a        ; If >0, Sqrxz has pulled a lever. 0 = not pulled
 ld (practice),a       ; Pratice mode off
 ld a,3
 ld (lives),a	       ; And start with 3 lives

 call ChangeOptions    ; Call to the option menu
 call ChooseWorld      ; And a call to select world to play on
 im 2		       ; Activate new interrupt handler

PlayWorld:
 ld a,1
 ld (gaming),a	       ; Now the game has started
 call _clrLCD	       ; Prepare to show world and level info
 ld hl,$FC00
 ld bc,127	       ; 128 bytes to be filled (BC=no-1)
 ld (hl),$FF
 call FillChar	       ; Fill the first 8 rows with black ((127+1)/16)=8
 ld hl,0
 ld (_curRow),hl       ; Set cursor position to top left corner
 ld hl,tmpWorld        ; HL now points to World data
 ld a,(hl)	       ; A = format; 0=uncompressed; 1=compressed
 ld (worldT),a	       ; Save it for later
 inc hl
 inc hl 	       ; Skip this byte, it's always 0 (Fargo compatibility)
 ld a,(hl)	       ; A = number of levels in this world
 ld (noLevels),a       ; Store that in the memory as well
 inc hl 	       ; HL -> name of world
 set textInverse,(iy+textflags)
 call _puts	       ; Display the name of the world
 res textInverse,(iy+textflags)
 ld a,101	       ; A = 128-the size of the text "made by "
 sub (hl)	       ; Subtract with the size of the author
 inc hl 	       ; HL -> author
 srl a		       ; Divide A with 2. This will center the text.
 ld d,$09
 ld e,a
 ld (_penCol),de       ; Set cursor (menustyle) to row 9, col A
 push hl
 ld hl,WorldAuthor     ; HL -> "made by " string
 call _vputs	       ; Show it
 pop hl
 call _vputs	       ; And show the name of the author directly after
 ld (lvlptrs),hl       ; Save HL for later, which points to level data

 ld a,(practice)
 or a		       ; Check if practice mode
 jr z,ShowLevelLives
 ld hl,$0403
 ld (_curRow),hl
 ld hl,PracticeTxt
 call _puts
 push hl
 ld hl,$FF90
 ld bc,112
 ld (hl),$FF
 call FillChar
 ld hl,$390C
 ld (_penCol),hl
 pop hl
 set textInverse,(iy+textflags)
 call _vputs
 res textInverse,(iy+textflags)
ShowLevel:
 ld hl,$0B05
 ld (_curRow),hl       ; Set cursor position to middle of screen
 ld a,(lvl)
 inc a		       ; Increase the level number since Level 1 really is 0
 call DispA	       ; And show the level number
 ld a,$06
 ld (_curCol),a
 ld hl,LevelTxt
 call _puts
SelectLevel:
 call _getky
 cp _kyEnter
 jr z,LevelSelected
 cp _kyExit
 jp z,QuickQuit
 ld hl,lvl
 ld de,noLevels
 dec a
 jr z,DecLevel
 cp 3
 jr nz,SelectLevel
 inc (hl)
 ld a,(de)
 cp (hl)
 jr nz,LevelChgd
 ld (hl),0
 jr LevelChgd
DecLevel:
 ld a,(hl)
 dec (hl)
 or a
 jr nz,LevelChgd
 ld a,(de)
 dec a
 ld (hl),a
LevelChgd:
 xor a
 ld (start_x),a
 jr ShowLevel

ShowLevelLives:
 ld hl,$0B07
 ld (_curRow),hl       ; Set cursor position to middle of screen
 ld a,(lvl)
 inc a		       ; Increase the level number since Level 1 really is 0
 call DispA	       ; And show the level number
 ld a,$06
 ld (_curCol),a
 ld hl,LevelTxt
 call _puts	       ; Display "Level"
 ld hl,$0A04
 ld (_curRow),hl
 ld a,(lives)
 call DispA	       ; Show lives left
 ld a,$0A
 ld (_curCol),a        ; Update the X coordinate
 ld a,'x'
 call _putc	       ; Put out the 'x' (must be done in this order)
 ld hl,Sqrxz	       ; HL -> Sqrxz sprite
 ld de,$FDF6	       ; DE -> position in video mem where it should be put
 ld b,8 	       ; 8 rows
PutSq:
 push bc
 ldi		       ; Shorter than ld a,(hl) \ ld (de),a \ inc hl
 ex de,hl
 ld bc,15	       ; Offset to next row (ldi increase HL and DE with 1)
 add hl,bc
 ex de,hl	       ; DE now points to the row below
 pop bc
 djnz PutSq	       ; And show the rest of the rows
WaitKey:
 call _getky
 cp 5		       ; Wait until a key except the arrowkeys has been pressed
 jr c,WaitKey	       ; This is necessary, because you could miss this screen
 cp _kyExit	       ; If exit was pressed
 jp z,QuickQuit        ;  then abort the game

LevelSelected:
 ld hl,LevelPtr        ; HL -> Future storage of temporary level
 ld bc,2047	       ; Prepare clear 2k memory. Max lev size is 255*8=2040
 ld (hl),0
 call ClearMem	       ; Clear it. Last 8 bytes are necessary to be cleared

 ld hl,(lvlptrs)       ; HL -> level info
 ld d,0 	       ; This saves a few bytes (ld de,xx won't be necessary)
 ld a,(worldT)
 or a		       ; Check if the world was compressed or not
 ld a,(noLevels)       ; A = number of levels (will be used in both cases)
 jr z,UncompWorld      ;  If uncompressed, the data is stored differently
 add a,a	       ; Mul noLevels with 4 to find the beginning of the
 add a,a	       ; huffman compressed data
 ld e,a 	       ; The D is already cleared, so DE=offset to skip
 push hl	       ; But HL must preserved
 add hl,de	       ; Now HL -> hufflib data
 ex (sp),hl	       ; Push that addr to the stack while popping the prev HL
 ld a,(lvl)
 ld b,a 	       ; B=level number='file' number in the huffman archive
 add a,a
 add a,a
 ld e,a
 add hl,de	       ; HL -> levelsize (word) and time (word)
 inc hl 	       ; Levelsize is always<256, thus this byte=0 (Big Indian)
 ld a,(hl)
 ld (LevelSize),a      ; Save the level size
 inc hl 	       ; HL -> time (word)
 ld d,(hl)
 inc hl
 ld e,(hl)	       ; DE = time to complete the level
 ld (timeleft),de      ; Save it
 pop hl 	       ; HL -> hufflib compressed data
 ld de,LevelPtr        ; DE -> where to store the data (temp level storage)
 ld ix,$8000	       ; IX -> 1024 bytes of free memory to be used by hufflib
 call HuffExtr	       ; Extract 'file' B with the above parameters
 jr LoadL	       ; The level is extracted, initialize the first part
UncompWorld:
 add a,a	       ; Multiply noLevels with 2
 ld e,a
 push hl	       ; Save HL for later (-> array of offsets)
 add hl,de	       ; HL -> start of level data (incl level info)
 ex (sp),hl	       ; Exchange stack contents with HL
 ld a,(lvl)
 add a,a	       ; Multiply level with two,
 ld e,a
 add hl,de	       ; thus getting the offset to the offset pointer
 ld d,(hl)
 inc hl
 ld e,(hl)	       ; Read that offset
 pop hl
 add hl,de	       ; Add it with the pointer to raw level data
 inc hl 	       ; Increase with 1. HL -> level to play info
 ld a,(hl)	       ; which starts with the level size
 ld (LevelSize),a
 inc hl 	       ; HL -> time to complete level
 push hl
 ld h,0
 ld l,a 	       ; Prepare to calculate how many bytes to copy
 add hl,hl
 add hl,hl	       ; Multiply level size with 8, since there 8 tiles
 add hl,hl	       ; in one column
 ex (sp),hl	       ; Exchange stack contents
 ld d,(hl)
 inc hl
 ld e,(hl)	       ; Read the time
 ld (timeleft),de      ; and store it
 inc hl 	       ; HL -> raw level data
 ld de,LevelPtr        ; DE -> temporary level storage
 pop bc 	       ; BC = bytes to copy (levelsize*8)
 ldir		       ; And copy it
LoadL:
 call LoadLevel        ; Initialize the current level
 call PutSqrxz	       ; Put Sqrxz on start position...

; The mainloop contains many steps which are (in order):

;  1. Animate all tiles that should, according to the AnimTable.
;  2. Check if Sqrxz is doing his death jump. If so, update him.
;  3. Check if Sqrxz is jumping. If so, try move him up, and skip next step.
;  4. Check if there is ground below him. If there isn't, fall down.
;  5. Check if there is bridge below Sqrxz. If there is, start bridge anim.
;  6. Check the down key. If it is held down, don't check other keys
;     if the last bit in the counter is set (jump to step 9).
;  7. Check the up key. Start the jump if possible.
;  8. Check the left and right keys, and move Sqrxz if possible.
;  9. Check the square(s) Sqrxz is/are at for portcullis, spikes, traps.
; 10. A short check if a portcullis should close.
; 11. Move all enemies if 'even' frame
; 12. Check if the enemies have to be removed from the screen to avoid trash.
;     This is the case if 1) the enemies have moved 2) the screen scrolls
;     or 3) Sqrxz is overlapping the enemy which happens when a) Sqrxz is
;     dying (death jump), b) when the Bat dies or c) when the green man is
;     hiding in the helemt
; 13. Remove Sqrxz from screen (if he moved), scroll screen if necessary,
;     put Sqrxz on the screen again and also put all enemies if they were
;     removed (see above).
; 14. Update the timebar (decrease) and the waterbar (decrease if in water,
;     else decrease).
; 15. If using Virtual Screen, flip page.
; 16. Then make a delay, so the game goes at a constant speed (if possible)
; 17. If Sqrxz is alive and the player hasn't aborted the game, jump to step 1.

MainLoop:
 xor a		       ; A lot of vars should be cleared each frame
 ld (timer),a
 ld (flags),a
 ld hl,counter
 inc (hl)	       ; Increase the overall counter
 ld hl,animcnt
 inc (hl)	       ; Increase the anim counter, which is modulo 3
 ld a,(hl)
 cp 3
 jr nz,CheckDead       ; Only animate 1/3
 ld (hl),0	       ; Clear the anim counter
 ld hl,AnimTable       ; HL -> animation table
 ld b,8 	       ; Max 8 animations at the same time
RepCheckAnim:
 push bc
 ld b,(hl)	       ; b = x coord
 inc hl
 push hl
 ld a,(hl)	       ; a = y coord
 cp 255 	       ; If 255 then no animation
 jr z,NextAnim
 inc hl
 ld e,(hl)
 inc hl
 ld d,(hl)	       ; DE loaded with (HL)
 ex de,hl	       ; HL = address in level to animating object
 ld c,a 	       ; B,C = coordinates
 ld a,(hl)	       ; Check which the new object should be
 cp 99		       ; A lever being pulled?
 jr z,NextAnimStep     ;  Pull it down
 cp 14		       ; Portcullis halfway down?
 jr z,NextAnimStep     ;  Then next should be port down
 cp 96		       ; Spikes halfway up?
 jr z,NextAnimStep     ;  Then next should be spikes up
 and %11111100	       ; Mask off the two last bits
 cp 20		       ; Check if A was 20-23 (Steel brick animation)
 jr nz,NA1	       ;  If not, check other
 ld a,(hl)
 inc a
 ld (hl),a	       ; Increase so it becomes the next animation tile
 cp 24		       ; Check if end of steel brick animation
 jr nz,ShowChange      ;  Not end - show the change
 ld a,2 	       ; End of animation - reset to steel brick
 jr EndOfAnim	       ; Mark end of animation
NextAnimStep:
 inc a		       ; Increase to the next animation step
 jr EndOfAnim	       ; EOA
NA1:
 cp 24		       ; Check if A was 24-27 (Brick crush animation)
 jr nz,NA2	       ;  Nope - check other
 inc (hl)	       ; Increase so it becomes the next animation tile
 ld a,(hl)
 cp 28		       ; Check if EOA
 jr nz,ShowChange      ;  Nope - show change
 xor a		       ; Reset to space
 jr EndOfAnim	       ; And EOA
NA2:
 cp 28		       ; Last check - was A 28-31 (Bridge animation)
 jr nz,NextAnim        ; No, skip to next entry (this jump shouldn't occur)
 inc (hl)	       ; Increase so it becomes the next animation tile
 ld a,(hl)
 cp 32		       ; Check if EOA
 jr nz,ShowChange      ;  Nope - show change
 xor a		       ; Reset to space
EndOfAnim:
 ld (hl),a	       ; Store the change in memory
 pop hl 	       ; HL -> that animation table entries Y addr
 ld (hl),255	       ; Store 255 which means no animation
 push hl
ShowChange:
 call CoordConv        ; Get where on screen that tile is
 cp 100 	       ; Is it the lever animation
 jr z,LeverAnim        ;  If so, special occasion
 call PutTileNo        ; And put tile A on that position
 jr NextAnim
LeverAnim:
 call RemoveSqrxz      ; Sqrxz needs to be removed because of trash
 ld a,100
 call PutTileNo
 call PutSqrxz	       ; Restore Sqrxz
NextAnim:
 pop hl 	       ; Restore HL
 inc hl
 inc hl
 inc hl 	       ; Increase HL with 3 so it points to the next entry
 pop bc 	       ; Pop the counter
 djnz RepCheckAnim     ; And check for new animation

CheckDead:	       ; This updates Sqrxz "death jump"
 ld a,(deadcnt)
 or a		       ; Check if Sqrxz is dead and has disappeard from screen
 jp nz,CheckEnemies    ; If so, skip a lot of stuff and update enemies
 ld hl,dead
 ld a,(hl)
 or a		       ; Check if dead
 jr z,CheckJump        ;  No - check other stuff
 inc a		       ; Increase the dead counter
 ld (hl),a
 ld hl,ArcData	       ; HL -> The precalculated "gravity" table
 ld d,0
 sub 22
 jr nc,DDown	       ; If counter>=22, Sqrxz is falling down
 inc a
 neg
 ld e,a
 add hl,de	       ; Add with offset
 ld a,(hl)	       ; Now A = no pixels to move Sqrxz up
 neg		       ; Make that negative since up is negative
 jr DMoveVert	       ; Do the vertical move
DDown:
 cp 27		       ; The counter has a limit.
 jr c,DDown2	       ; If above,
 ld a,26	       ;  set it to max
DDown2:
 ld e,a
 add hl,de	       ; Add with offset
 ld a,(hl)	       ; Now A = no pixel to move Sqrxz (down)
DMoveVert:
 ld hl,y
 add a,(hl)	       ; Add the Y coordinate with A
 ld (hl),a
 ld hl,(x)
 ld a,(sqrxzdir)       ; The horizontal death move is made in the opposite
 or a		       ; direction of which Sqrxz faces
 jr z,DLeft
 inc hl 	       ; If facing left, increase X coordinate
 jr PutDeadSqrxz
DLeft:
 dec hl 	       ; Else decrease it
PutDeadSqrxz:
 ld (x),hl	       ; Store the change
 ld hl,flags
 set 2,(hl)	       ; Set a flag telling that Sqrxz has moved this frame
 jp CheckEnemies       ; Check if enemies are to move - skip the user control

CheckJump:
 ld a,(jump)	       ; Check the jump counter
 or a		       ; Is it 0 (no jump in progress)?
 jr z,CheckFall        ; Yup - check if falling (or if Sqrxz should fall)
 ld hl,jump
 dec (hl)	       ; Decrease the jump counter (it counts backwards)
 ld a,(hl)
 or a
 jr z,CheckFall        ; If 0, the jump is over. Check if he should start fall
 ld hl,ArcData	       ; HL -> The precalculated "gravity" table
 ld d,0
 ld e,a
 add hl,de	       ; Add with the offset (the jumping counter)
 ld a,(hl)	       ; Now A = number of pixels to move Sqrxz up this frame
 or a		       ; If no pixels,
 jp z,CheckBridge      ;  skip it for this time and skip the fall checking
 ld b,a 	       ; Load the counter register with number of steps
RepJump:
 push bc
 call MoveUp	       ; Try to move up one pixel
 pop bc
 jr c,HitHead	       ; Ouch! Something is in the way - check what it is
 djnz RepJump	       ; Else repeat the loop
 call CheckUp	       ; Check if anything above
 jp nc,CheckBridge     ;  If not, end of jump routine
HitHead:
 xor a
 ld (jump),a	       ; Clear the jump counter - the jump is over
 ld a,(y)
 dec a		       ; A = y position of tile above
 ld hl,(x)
 ld de,4
 add hl,de	       ; HL = Sqrxz x position+4 (middle of Sqrxz)
 call GetBlock	       ; Find out what tile is above
 dec a		       ; Is it a brick wall (1)?
 jr z,HeadInBrick      ;  Yes - start a Brick crush animation
 dec a		       ; Is it a steel brick (2)?
 jr nz,CheckFall       ;  Nope - check if Sqrxz should start fall instead
 ld a,21	       ; It was a steel brick above - start an animation
NewA:
 call NewAnim	       ; Add a new animation (A) to the AnimTable
 jr CheckFall	       ; And check if Sqrxz should start fall
HeadInBrick:
 ld a,24	       ; Prepare to start a Brick Crush animation
 jr NewA	       ; Start it

CheckFall:
 call CheckDown        ; Try to move Sqrxz down one step
 jp c,NoFall	       ;  There's ground below! Don't fall
 ld hl,fall	       ; No ground below,
 inc (hl)	       ; increase the fall counter
 ld a,(hl)
 cp 27		       ; The counter ranges from 0-26
 jr nz,FallDown        ; (you can't fall faster than a max value)
 ld a,26	       ; If >26, set to 26
 ld (hl),a
FallDown:
 ld hl,ArcData	       ; HL -> precalculated "gravity" table
 ld d,0
 ld e,a
 add hl,de	       ; Add with offset (the falling counter)
 ld a,(hl)	       ; Now A = number of pixels to move Sqrxz down this frame
 or a		       ; If 0 pixels,
 jp z,CheckBridge      ;  no moving down this frame
 push af
 ld c,a
 ld ix,enemytable      ; IX -> enemy table
 ld b,8 	       ; Max 8 enemies to check
CheckEnemyKill:
 ld a,(ix)
 or a		       ; Check if there is an enemy at this table entry
 jr z,CEKNext	       ;  If not, check next entry
 cp 2		       ; Check if hedgehog (can't kill those)
 jr z,CEKNext	       ;  If yes, then check next entry
 cp 5		       ; Check if fish (can't kill those either)
 jr z,CEKNext	       ;  If yes, then check next entry
 ld a,(ix+6)
 or a		       ; Check if something is happening to the enemy
 jr nz,CEKNext	       ;  If so, then he can't be killed
 ld hl,(x)
 ld d,(ix+3)
 ld e,(ix+2)	       ; DE = enemy x coordinate
 sbc hl,de	       ; Subtract Sqrxz X coordinate with the enemies
 ld a,l
 jr nc,AbsX	       ; If the result <0
 neg		       ;  then negatate it (the absolute result is important)
AbsX:
 cp 8		       ; If the difference between the coordinates>=8
 jr nc,CEKNext	       ;  Then no kill is possible - check next entry
 ld a,(ix+4)
 ld hl,y
 sub (hl)	       ; Subtract enemies Y coord with Sqrxz
 sub 8		       ; Subtract with 8 (the sprite size)
 bit 7,a	       ; Check if result <0
 jr nz,CEKNext	       ;  If so, Sqrxz is not above enemy - check next entry
 sub c		       ; Subtract with number of steps to fall
 jr nc,CEKNext	       ;  If not carry, then Sqrxz doesn't fall enough
 ld (ix+6),1	       ; Set a flag indicating a hit
 xor a
 ld (fall),a	       ; Clear fall counter
 ld a,(ix+4)
 sub 8
 ld (y),a	       ; Set Sqrxz Y coordinate equal to enemy-8
 pop bc 	       ; Just removing the unncessary data from the stack
 ld a,6
 ld (jump),a	       ; A small jump is made when hitting an enemy
 ld a,(jump_button)
 or a		       ; Has the jump button been released since last jump?
 jr nz,CheckBridge     ;  If not, don't check if jump button is pressed
 ld a,$FE
 out (1),a
 nop
 nop
 in a,(1)
 bit 3,a	       ; Check if the up key is being pressed
 jr z,JumpOnEnemy      ;  If so, start a jump
 ld a,$BF
 out (1),a
 nop
 nop
 in a,(1)
 bit 5,a	       ; Check if the 2nd key is being pressed
 jr nz,CheckBridge     ;  If not, do other stuff
JumpOnEnemy:
 ld a,20
 ld (jump),a	       ; Else set the jumpcounter to 20 (max value)
 ld a,1
 ld (jump_button),a    ; And set the flag indicating jump button is down
 jr CheckBridge
CEKNext:
 ld de,7
 add ix,de	       ; Add with 7 to get the next entry
 djnz CheckEnemyKill   ; And loop
 pop bc 	       ; Pop the counter register with no steps
RepFallDown:
 push bc
 call MoveDown	       ; Move down one step. No checking for ground below
 pop bc 	       ; is needed, because the way the "gravity" table is made
 djnz RepFallDown      ; Repeat the falling
 jr CheckBridge        ; And check some other stuff
NoFall:
 xor a
 ld (fall),a	       ; Reset the fall counter

CheckBridge:	       ; This routine checks if there is a bridge below you
 ld a,(y)
 and 7		       ; Check if the last three bits in Y is 0.
 jr nz,CheckKeys       ;  Nope, Sqrxz is between (in y dir) two tiles
 call CheckDown        ; Get the two tiles below Sqrxz. B = leftD, C = rightD
 ld a,b
 cp 16		       ; Check if bridge below Sqrxz to the left
 jr z,BridgeBelow      ;  Yup - check if it should start animating
 ld a,c
 cp 16		       ; Else check if bridge below Sqrxz to the right
 jr nz,CheckKeys       ;  No, no bridge. Check if user has pressed any keys
BridgeBelow:
 ld a,b 	       ; Since B or C = 16 (bridge), and if B+C=16, it would
 add a,c	       ; mean that Sqrxz his his centre of gravity on the
 cp 16		       ; bridge (since B or C would be 0 then)
 jr nz,CNearB	       ;  No, that wasn't the case - check where CoG is
 ld hl,(x)	       ; HL = Sqrxz X coordinate
 ld a,b
 or a
 jr nz,BGetB	       ; Check if it was B=16. If so find out what's below CoG
 ld de,7
 add hl,de	       ; Make HL point to Sqrxz's right side
 jr BGetB	       ; And find out what tile is below there
CNearB:
 ld hl,(x)	       ; So, there is a bridge below combined with something
 inc hl 	       ; else. The centre of gravity is in the middle of
 inc hl 	       ; Sqrxz (since both tiles are ground tiles), so add
 inc hl 	       ; HL with 4 so it contains the X position to middle
 inc hl 	       ; of Sqrxz
BGetB:		       ; Here, HL = Sqrxz Centry Of Gravity (xdir)
 ld a,(y)
 add a,8	       ; A = y location below Sqrxz
 call GetBlock	       ; Find out what block is below
 cp 16		       ; Check if it is a bridge
 jr nz,CheckKeys       ; If not, the CoG stands on a brick or earth - no anim
BridgeBroken:
 ld a,28	       ; Else that bridge should start animating (crashing)
 call NewAnim	       ; Start a new animation (28 = first bridge crush anim)

CheckKeys:
 ld a,$BF	       ; Set the mask (scanning for 2nd)
 out (1),a	       ; Send the mask to the port
 nop
 nop
 in a,(1)	       ; And receive the "anser"
 and $F0	       ; Mask unnecessary bits
 ld c,a
 ld a,$FE	       ; Set the mask
 out (1),a	       ; Send the mask to the port
 nop
 nop
 in a,(1)	       ; And receive the "anser"
 bit 0,a	       ; Check if the down key is held down
 jr nz,CheckMovement   ;  If not, check for movement
 ld hl,counter	       ; When the down key is held down, you only move
 bit 0,(hl)	       ; if the main counter is an even number - check that
 jr nz,CheckSquare     ; No it was an odd number - skip movement checking

CheckMovement:
 push af
 bit 3,a	       ; Is the up key held down?
 jr z,TryJump	       ;  Yup - check if it's possible to jump
 bit 5,c	       ; Is the 2nd key held down?
 jr z,TryJump	       ;  Yup - check if it's possible to jump
 xor a
 ld (jump_button),a    ; Clear the "jump button not pressed" flag
 jr CheckHorzKeys      ; Check for horizontal movement
TryJump:
 call StartJump        ; Start the jump is possible
CheckHorzKeys:
 pop af

 push af
 bit 1,a	       ; Left key held down?
 call z,MoveLeft       ;  Yup - move left if possible
 pop af
 bit 2,a	       ; Right key held down?
 call z,MoveRight      ;  Yup - move right if possible
 ld hl,flags
 bit 2,(hl)	       ; Check if Sqrxz has moved this frame
 jr nz,MovedHor        ;  Yes Sqrxz has moved
 ld a,(hmovelast)
 or a		       ; Did Sqrxz move last frame?
 jr z,CheckSquare      ;  Nope - check some other stuff
 ld hl,sqrxzcnt        ; Prepare to clear a bit in the Sqrxz anim counter
 res 1,(hl)	       ; This will prevent Sqrxz from having his legs
 xor a		       ; in the air if he's not moving (would look stupid)
 ld (hmovelast),a      ; Clear the flag - Sqrxz didn't move this frame
 jr SetUpdateSqrxz     ; Sqrxz is to be updated (the Sqrxz animation changed)

MovedHor:
 ld a,1
 ld (hmovelast),a      ; Set the Sqrxz Moved Last Frame flag
SetUpdateSqrxz:
 ld hl,flags
 set 2,(hl)	       ; And set the Sqrxz Moved This Frame flag

CheckSquare:
 ld a,(y)
 add a,3	       ; A = y coordinate at Sqrxz feet
 ld hl,(x)	       ; HL = Sqrxz x coordinate
 call GetBlock	       ; Find out what block is to the left of Sqrxz
 cp 93		       ; Extra life?
 call z,ExtraLife      ;  Increase life
 cp 64		       ; Sausage?
 call z,EatSausage     ;  Eat it
 cp 98		       ; Lever?
 call z,PullLever      ;  Pull it
 cp 97		       ; Visible spike?
 jr z,SqrxzDied        ;  DIE!
 cp 94		       ; An open portcullis?
 call z,PCFound        ;  Yup - prepare for it to fall down
 cp 95		       ; A spike trap?
 jr nz,CSNext	       ;  No - check for block where Sqrxz is to the right
 ld a,96	       ; Prepare to start a spike animation
 call NewAnim	       ; Start it
 ld hl,flags
 set 3,(hl)	       ; Set a flag indicating Sqrxz died this frame
CSNext:
 ld a,(y)
 add a,3	       ; A = y coordinate at Sqrxz feet
 ld hl,(x)
 ld de,7
 add hl,de	       ; HL = Sqrxz x coordinate+7 (the right part of him)
 call GetBlock	       ; Find out what block is to the right of Sqrxz
 cp 104 	       ; Is it the bottomright corner of an exit door?
 jp z,LevelFinished    ;  YES! Level finished!
 cp 93		       ; Extra life?
 call z,ExtraLife      ;  Increase life
 cp 64		       ; Sausage?
 call z,EatSausage     ;  Eat it
 cp 98		       ; Lever?
 call z,PullLever      ;  Pull it
 cp 97		       ; Visible spike?
 jr z,SqrxzDied        ;  DIE!
 cp 94		       ; An open portcullis?
 call z,PCFound        ;  Yup - prepare for it to fall down
 cp 95		       ; A spike trap?
 jr nz,CheckIfDied     ;  No - check if Sqrxz has died this frame
 ld a,96	       ; Prepare to start a spike animation
 call NewAnim	       ; Start it
 ld hl,flags
 set 3,(hl)	       ; Set a flag indicating Sqrxz died this frame

CheckIfDied:
 ld hl,flags
 bit 3,(hl)	       ; Check if Sqrxz died this frame
 jr z,CheckPort        ;  No, he didn't - check if a portcullis should close
SqrxzDied:
 call KillSqrxz

CheckPort:
 ld hl,(portc)	       ; HL = x,y of portcullis to close
 ld a,h
 or l		       ; Check if HL=0
 jr z,CheckEnemies     ;  It was - no such port. Check for enemies
 ld hl,flags
 bit 4,(hl)	       ; Check if Sqrxz was below a portcullis this frame
 jr nz,CheckEnemies    ;  Yes he was - can't close the portcullis now.
 ld hl,portc
 xor a
 ld b,(hl)	       ; B = x coor of portcullis to go down
 ld (hl),a	       ; Clear that byte
 inc hl
 ld c,(hl)	       ; C = y coor of portcullis to go down
 ld (hl),a	       ; Clear that byte as well
 inc hl
 call _ldhlind	       ; HL = address in level where the portcullis is
 ld a,14	       ; Prepare to start a closing portcullis animation
 call NewAnim	       ; Start it

CheckEnemies:
 call MoveEnemies      ; Move all active enemies
 ld a,(dead)
 or a		       ; Check if dead
 jr nz,NoScroll        ;  Never scroll if Sqrxz is doing his death jump
 ld hl,(x)
 ld de,(XScr)
 or a
 sbc hl,de	       ; HL = Sqrxz X position on screen
 ld a,l
 cp 85		       ; Check if >=85
 jr nc,Scroll	       ;  If so, scroll
 cp 70		       ; Check if <70
 jr c,NoScroll	       ;  If so, no scroll
 ld hl,counter
 bit 0,(hl)	       ; If between 70-84, scroll every second frame
 jr z,NoScroll	       ; If an "even" frame, skip scrolling
Scroll:
 call RemoveEnemies    ; Must remove enemies before scrolling
 call ScrollLeft       ; Scroll screen!
NoScroll:
 ld hl,counter
 bit 0,(hl)	       ; Check if it was an even frame
 jr z,ScanEnemies      ;  Yes, no enemy moves this frame
 call RemoveEnemies    ; Else remove the enemies
 jr CheckSqUpdate      ; Skip ScanEnemies since they have already been removed

ScanEnemies:	       ; Check if Sqrxz is close to an enemy
 ld ix,enemytable      ; IX -> enemy table
 ld b,8 	       ; 8 entries to check
ScanEnemy:
 ld a,(ix)
 or a		       ; Is there an enemy at this entry?
 jr z,ScanNext	       ;  Nope, check next entry
 ld a,(y)
 ld c,(ix+4)
 sub c		       ; Now A = difference between the y coordinates
 bit 7,a
 jr z,SE_AbsY
 neg		       ; If A<0, then A=-A
SE_AbsY:
 cp 10
 jr nc,ScanNext        ; If A>=10 then enemy is not close
 ld d,(ix+3)
 ld e,(ix+2)
 ld hl,(x)
 or a
 sbc hl,de	       ; Now HL = difference between the x coordinates
 ld a,l 	       ; A = dif
 jr nc,SE_AbsX	       ; If negative,
 neg		       ;  then make it positive
SE_AbsX:
 cp 10
 jr nc,ScanNext        ; If A>=10 then enemy is not close
 call RemoveEnemies    ; Remove all enemies
 jr CheckSqUpdate
ScanNext:
 ld de,7
 add ix,de	       ; Point to next enemy entry
 djnz ScanEnemy        ; Scan next enemy
CheckSqUpdate:
 ld hl,flags
 bit 2,(hl)	       ; Check if Sqrxz has moved this frame
 jr z,NoSqMove	       ;  If not, don't update Sqrxz (would cause flicker)
 call RemoveSqrxz      ; Remove Sqrxz from previous location
 call PutSqrxz	       ; Put Sqrxz on new location
NoSqMove:
 ld hl,flags
 bit 5,(hl)	       ; Check if RemoveEnemies has been called
 call nz,PutEnemies    ;  Yes, put the enemies on the screen again

 ld hl,(x)	       ; Now check if Sqrxz nose is in water
 ld a,(sqrxzdir)
 or a
 jr nz,SqLeft	       ; If Sqrxz is faced to the right,
 ld de,7	       ;  the nose is to the right part of the sprite,
 add hl,de	       ; thus adding 7 to the x coodrinate
SqLeft:
 ld a,(y)
 add a,5	       ; Always add 5 to the y coordinate
 call GetBlock	       ; Find out what tile is at that location
 ld hl,(waterbar)      ; HL=water bar
 cp 109 	       ; Check if tile is <109
 jr c,IncreaseWaterBar ;  If so, increase the water bar
 dec hl 	       ; Else decrease it
 ld (waterbar),hl      ; Save the new value
 ld a,l
 or h		       ; Check if HL has reached zero
 jp z,Drowned	       ;  If so, Sqrxz has drowned :(
 and $F8
 ld l,a 	       ; This will clear the last three bits in HL
 add hl,hl	       ; Mul HL with two
 ex de,hl	       ; DE = offset where pixel should be removed
Modify_7:
 ld hl,$FFFF	       ; Subtract from end of videomem
 sbc hl,de	       ;  (the carry was cleared by the AND instruction above)
 res 4,(hl)	       ; Reset bit 4 at that address, which is the water bar
 jr UpdateTime	       ; Jump to the timer update routine
IncreaseWaterBar:
 ld de,479
 call _cphlde	       ; Check if top of bar is reached
 jr z,UpdateTime       ;  If so, don't increase the bar
 push hl	       ; The old value is the one to be deleted later
 inc hl 	       ; Increase bar
 call _cphlde	       ; Check if top reached
 jr z,UpdateWaterBar   ;  If so, don't increase another step
 inc hl 	       ; Increase again - the bar increase faster than decr
UpdateWaterBar:
 ld (waterbar),hl      ; Update the new value
 pop hl 	       ; HL = old water bar value
 ld a,l
 and $F8
 ld l,a
 add hl,hl	       ; DE = offset where pixel should be added
 ex de,hl
Modify_8:
 ld hl,$FFFF
 sbc hl,de	       ; HL -> the correct location in the video memory
 set 4,(hl)	       ; And set the pixel
UpdateTime:
 ld hl,(timeleft)
 dec hl 	       ; Always decrease the time with two - the time "format"
 dec hl 	       ; is from the Fargo version
 ld (timeleft),hl      ; Save the updated time
 ld a,h 	       ; A = time/256
 or l		       ; Check if out of time
 jr z,TimeOut	       ;  If so, time is out - show message
 add a,a
 add a,a
 ld h,0
 ld l,a
 add hl,hl
 add hl,hl	       ; HL = time/16, but with last 4 bits cleared
 ex de,hl

Modify_1:
 ld hl,$FFFF	       ; Prepare to find where in video mem to clear
 sbc hl,de	       ; (carry was cleared by add hl,hl)
 ld a,$F0
 and (hl)	       ; Mask the last four bits, which is the timer
 ld (hl),a	       ; Update changes to videomem
Modify_B:
 jr SkipFlip	       ; Skip the flipping (will be selfmodified if virtscr)
 ld hl,_plotSScreen    ; HL -> virtual screen (graph memory)
 ld de,$FC00	       ; DE -> video memory
 ld bc,1024	       ; BC = screen size
 ldir		       ; Copy screen
SkipFlip:
 ld hl,timer	       ; HL -> the interrupt timer
Wait:
 ld a,(hl)
speedC:
 cp 3		       ; Has it gone 3/200 sec since frame started?
 jr c,Wait	       ;  Nope, wait until it has
 ld a,$BF
 out (1),a
 nop
 nop
 in a,(1)
 bit 6,a	       ; Check if Exit has been pressed
 jr z,LoseLife	       ;  If so, quit the game
 bit 7,a	       ; Check if More has been pressed
 jp z,Pause	       ;  If so, pause the game
PauseResume:
 ld a,(y)	       ; Prepare to find out if Sqrxz y coordinate is >=64
 cp 150 	       ; If it's bigger then 150, then it's negative
 jp nc,MainLoop        ;  It was >150, restart frame from beginning
 cp 64		       ; Else check if >=64
 jp c,MainLoop	       ;  Nope it wasn't. Sqrxz is alive - jump to mainloop

 call _getky
 or a		       ; Check if any key has been pressed
 jr nz,LoseLife        ; If so, skip the pause
 ld hl,deadcnt
 inc (hl)	       ; Increase death pause counter
 ld a,(hl)
 cp 200 	       ; Check if 200 frames has passed since Sqrxz disappeard
 jp c,MainLoop	       ; If not, repeat
LoseLife:
 call _getky	       ; Clear keybuffer
 xor a
 ld (sausages),a       ; Reset sausages counter
 ld a,(practice)
 or a
 jp nz,PlayWorld
 ld hl,lives
 dec (hl)	       ; Lose a life
 jp z,QuickQuit        ; If no lifes left, quit game
 jp PlayWorld	       ; And restart the level

TimeOut:
 ld hl,OutOfTimeTxt    ; HL -> "OUT OF TIME"
ShowDeathMess:
 ld de,$0404
 ld (_curRow),de       ; Set cursor position to middle of screen
 set textInverse,(iy+textflags)
 call _puts	       ; Show text
 res textInverse,(iy+textflags)
WaitK:
 call _getky
 cp 5
 jr c,WaitK	       ; Wait until a key (except arrowkeys) gets pressed
 jr LoseLife	       ; Jump to label above which decrease your life

Drowned:
 ld hl,DrownedTxt      ; HL -> "YOU DROWNED"
 jr ShowDeathMess      ; Same procedure as above, so use that code

LevelFinished:
 xor a
 ld (start_x),a        ; If you've pulled a lever, clear that
 ld a,(practice)
 or a
 jp nz,PlayWorld
 ld hl,lvl
 inc (hl)	       ; Increase the level
 ld a,(noLevels)
 cp (hl)	       ; Check if the previous level was the last one
 jp nz,PlayWorld       ;  If not, play this level
 call ShowTitle        ; Show title
 ld hl,$0302	       ; Prepare to display a congratulation message
 ld (_curRow),hl
 ld hl,GameFinTxt
 call _puts
 ld de,$0104
 ld (_curRow),de
 call _puts
 ld de,$0005
 ld (_curRow),de
 call _puts
 call WaitEnter
 jp QuickQuit

PCFound:
 ex de,hl	       ; Prepare to store info about the portcullis
 ld hl,portc
 ld (hl),b	       ; Store the x coordinate
 inc hl
 ld (hl),c	       ; Store the y coordinate
 inc hl
 ld (hl),e	       ; And finally store the address
 inc hl
 ld (hl),d
 ld hl,flags
 set 4,(hl)	       ; Set the flag "Sqrxz is below a portcullis"
 ret

EatSausage:
 push af	       ; The A register must be preserved when returning
 xor a
 ld (hl),a	       ; Remove the sausage from temp level storage
 push bc
 call RemoveEnemies    ; Removing enemies to avoid trash
 call RemoveSqrxz      ; And the same procedure with Sqrxz
 pop bc
 call CoordConv        ; Converts the coordinate to screen coordinates
 xor a
 call PutTileNo        ; Put an empty tile at those coordinates
 ld hl,sausages
 ld a,(hl)
 inc a		       ; Increase the number of sausages
 and 15 	       ; The counter is modulo 16
 ld (hl),a
 jr nz,NoExtraLife     ; If the counter didn't wrap, don't increase lives
 ld hl,lives
 inc (hl)	       ; Else increase number of lives
NoExtraLife:
 call PutSqrxz	       ; Put out Sqrxz again (the enemies will be done later)
 pop af
 ret

ExtraLife:
 push af	       ; The A register must be preserved when returning
 xor a
 ld (hl),a	       ; Remove the extra life from temp level storage
 push bc
 call RemoveEnemies    ; Removing enemies to avoid trash
 call RemoveSqrxz      ; And the same procedure with Sqrxz
 pop bc
 call CoordConv        ; Converts the coordinate to screen coordinates
 xor a
 call PutTileNo        ; Put an empty tile at those coordinates
 ld hl,lives
 inc (hl)	       ; Increase number of lives
 call PutSqrxz	       ; Put out Sqrxz again (the enemies will be done later)
 pop af
 ret

PullLever:
 push af
 ld (start_y),bc       ; Update the startcoord so Sqrxz starts here next life
 ld a,99
 call NewAnim	       ; Start the lever animation
 pop af
 ret

Pause:
 ld hl,$FC00
 ld de,_plotSScreen
 ld bc,1024
 ldir		       ; Copy screen to graph memory
 call ChangeOptions    ; Change options
 ld hl,_plotSScreen
 ld de,$FC00
 ld bc,1024
 ldir		       ; Restore screen
 jp PauseResume        ; Resume game

MoveLeft:
 call CheckLeft        ; Check if possible to move left
 ret c		       ;  Nope, it wasn't
 ld hl,(x)
 ld de,(XScr)
 call _cphlde	       ; Check if Sqrxz is at the left edge of screen
 ret z		       ;  Yes he is - then you can't move left
 dec hl 	       ; Decrease Sqrxz x coordinate
 ld (x),hl	       ; Store it
 ld hl,flags
 set 2,(hl)	       ; Set the "Sqrxz has moved" flag
 ld a,1
 ld (sqrxzdir),a       ; Change Sqrxz direction to Left
 ld hl,sqrxzcnt
 inc (hl)	       ; Update the Sqrxz animation counter
 ret

MoveRight:
 call CheckRight       ; Check if possible to move right
 ret c		       ;  Nope, it wasn't
 ld hl,(XScr)
 ld de,112	       ; (15-1)*8=112
 add hl,de
 ld de,(x)
 call _cphlde	       ; Check if Sqrxz has reached the right edge of screen
 ret z		       ;  Yes he has - then you can't move right
 inc de 	       ; Else increase the x coordinate
 ld (x),de	       ; And store it
 ld hl,flags
 set 2,(hl)	       ; Set the "Sqrxz has moved" flag
 xor a
 ld (sqrxzdir),a       ; Change Sqrxz direction to Right
 ld hl,sqrxzcnt
 inc (hl)	       ; Update the Sqrxz animation counter
 ret

StartJump:
 ld a,(fall)	       ; Now a procedure to check if Sqrxz is falling...
 ld hl,jump
 or (hl)	       ; ...or jumping...
 ld hl,jump_button
 or (hl)	       ; ...or haven't released the jump button since last jump
 ret nz 	       ; If that's the case, no jumping is allowed.
 ld a,20
 ld (jump),a	       ; Else set the jumpcounter to 20 (max value)
 ld a,1 	       ; Set the flag indicating that the jump button
 ld (jump_button),a    ; has been pressed this frame
 ret

MoveDown:
 call CheckDown        ; Check if it's possible to move down
 ret c		       ;  Return if not possible
 ld hl,flags
 set 2,(hl)	       ; Set the flag indicating that Sqrxz has moved
 ld hl,y
 inc (hl)	       ; Increase the y coordinate
 ret

MoveUp:
 call CheckUp	       ; Check if it's possible to move up
 ret c		       ;  Return if not possible
 ld hl,flags
 set 2,(hl)	       ; Set the flag indicating that Sqrxz has moved
 ld hl,y
 dec (hl)	       ; Decrease the y coordinate
 ret

MoveEnemies:
 ld hl,counter
 bit 0,(hl)	       ; Check if even frame number
 ret z		       ;  If so, don't move enemies this frame
 ld hl,(x)
 ld (tmp_x),hl	       ; Save Sqrxz coordinates into temporary position
 ld a,(y)	       ; The enemies will use the same detecting routines
 ld (tmp_y),a	       ; as Sqrxz, so they need to borrow Sqrxz variables
 ld ix,enemytable      ; IX -> at the beginning of the enemy table
 ld b,8 	       ; There can be at most 8 enemies moving at the same time
ControlEnemy:
 push bc
 ld a,(ix)	       ; Find out which enemy
 or a
 jp z,NextEnemy        ; If none, check next enemy
 ld a,(ix+6)
 or a
 jp nz,NextEnemy       ; If spec>0, the enemy is not moving at the moment
 ld h,(ix+3)
 ld l,(ix+2)	       ; HL = enemy x location
 ld (x),hl
 ld a,(ix+4)	       ; A = enemy y location
 ld (y),a
 ld de,(XScr)
 ld bc,64
 add hl,bc
 sbc hl,de	       ; Check if enemy is too far left of Sqrxz
 jr c,EnemyGone        ;  If so, the enemy will be 'gone'
 ld a,(LevelSize)
 ld h,0
 ld l,a
 add hl,hl
 add hl,hl
 add hl,hl
 ex de,hl
 ld hl,(x)
 call _cphlde	       ; Check if enemy has disappeared to the right of level
 jr nc,EnemyGone       ;  If so, enemy gone
 ld a,(ix)
 cp 3		       ; Check if bat
 jr z,EHorMove	       ;  If so, skip the vertical movement
 cp 5		       ; Check if fish
 jr z,EHorMove	       ;  If so, skip the vertical movement
 call CheckDown        ; Check if enemy can fall down
 jr c,EResetFall       ;  If it can't, reset fall counter
 ld a,(ix+5)
 inc a		       ; Increase fall counter
 cp 3
 jr nc,EFallS
 ld a,(ix)
 cp 4
 ld a,3 	       ; If the enemy is a green man, he should start
 jr nz,EFallS	       ; falling slower so he doesn't fall down into
 ld a,2 	       ; small holes
EFallS:
 cp 14
 jr nz,EFallDown       ; Can't fall faster than a certain value
 ld a,13
EFallDown:
 ld (ix+5),a	       ; Update the fall counter
 ld hl,EArc
 ld d,0
 ld e,a
 add hl,de
 ld a,(hl)	       ; Now A = number of steps to fall this frame
 or a
 jr z,EHorMove	       ; If no steps, check vertical movement
 ld b,a
EFalling:
 push bc
 call CheckDown        ; Try move down one step
 pop bc
 jr c,EStopFall        ; If not possible, stop the fall
 ld hl,y
 inc (hl)	       ; Increase y coordinate
 djnz EFalling	       ; Repeat the fall
EStopFall:
 ld a,(y)
 cp 64		       ; Check if Y>=64
 jr c,EHorMove	       ; If not, move enemy in horizontal direction
EnemyGone:
 ld (ix),0	       ; Clear current position in enemy table
 jr NextEnemy
EResetFall:
 ld (ix+5),0	       ; Resets the fall counter
EHorMove:
 ld b,1
 ld a,(ix)
 cp 4
 jr nz,ERepHorMove
 ld b,2 	       ; If green man, the enemy should move two pixels
ERepHorMove:
 push bc
 bit 0,(ix+1)	       ; Check direction
 jr z,EDirRight        ; If (ix+1)=0, the enemy should try right direction
 call CheckLeft        ; Check if possible to move left
 jr c,ELeftStop        ; If not, change direction
 ld hl,(x)
 dec hl 	       ; Decrease X coordinate
 ld (x),hl
 jr EHorMoveDone       ; Horizontal move done
ELeftStop:
 res 0,(ix+1)	       ; Change direction to right
 jr EHorMoveDone
EDirRight:
 call CheckRight       ; Check if possible to move right
 jr c,ERightStop       ; If not, change direction
 ld hl,(x)
 inc hl 	       ; Increase X coordinate
 ld (x),hl
 jr EHorMoveDone       ; Horizontal move done
ERightStop:
 set 0,(ix+1)	       ; Change direction to left
EHorMoveDone:
 pop bc
 djnz ERepHorMove      ; If green man, repeat once
UpdateEnemy:
 ld hl,(x)
 ld de,(tmp_x)
 or a
 sbc hl,de
 ld de,8
 call _cphlde
 jr c,CheckYDif
 dec de
 add hl,de
 jr nc,NoCollision
CheckYDif:
 ld a,(y)
 ld hl,tmp_y
 sub (hl)
 jr nc,AbsY
 neg
AbsY:		       ; Now A = abs(enemy_y-sqrxz_y)
 cp 8
 call c,KillSqrxz      ; If distance<8, set Sqrxz dead flag
NoCollision:
 ld de,(x)	       ; This will save (x) and (y) into the enemy table
 ld (ix+3),d
 ld (ix+2),e
 ld a,(y)
 ld (ix+4),a
NextEnemy:
 ld de,7
 add ix,de	       ; Nox IX -> the next enemy table entry
 pop bc
 dec b
 jp nz,ControlEnemy    ; Repeat until all enemies have been processed
 ld hl,(tmp_x)
 ld (x),hl	       ; Restore Sqrxz coordinates into the
 ld a,(tmp_y)
 ld (y),a	       ; "real variables"
 ret

KillSqrxz:
 ld hl,dead
 ld a,(hl)
 or a		       ; Check if Sqrxz is already dead
 ret nz 	       ;  If so, don't kill him again
 inc a
 ld (hl),a
 ld (sqrxzdir),a       ; Face Sqrxz left (temporary)
 ld hl,(x)
 ld de,(XScr)
 or a
 sbc hl,de	       ; HL = Sqrxz pos, screen relative
 ld a,l
 cp 64		       ; Check if X rel>=64
 ret c		       ;  Nope, don't change Sqrxz face direction
 xor a
 ld (sqrxzdir),a       ; Face Sqrxz right instead
 ret

NewAnim:	       ; Adds a new animation A at (B,C) - addr HL.
 ld (hl),a	       ; Store the tile no in the temp level storage
 push af
 ld a,(animptr)        ; Now A=next free entry in animation table
 push af
 push hl
 add a,a
 add a,a	       ; Multiply with 4 since each entry is 4 bytes
 ld d,0
 ld e,a
 ld hl,AnimTable
 add hl,de	       ; HL -> free location in animation table
 ld (hl),b
 inc hl
 ld (hl),c
 inc hl 	       ; X and Y coordinates have been stored
 pop de 	       ; DE = address in memory where the tiles is
 ld (hl),e
 inc hl
 ld (hl),d	       ; Now that address is stored as well
 pop af
 inc a		       ; Increase the animation pointer
 and 7		       ; Which is modulo 8
 ld (animptr),a        ; Update with the new value
 call CoordConv        ; Convert the coordinates of the tile
 pop af
 cp 99		       ; Check if it was a lever animation
 jr z,StartLever       ;  If so, Sqrxz needs to be removed
 call PutTileNo        ; Else just change the tile
 ret
StartLever:
 call RemoveSqrxz      ; Since Sqrxz overlaps the lever, he needs to be removed
 call PutTileNo        ; Put the tile
 call PutSqrxz	       ; And restore Sqrxz on screen
 ret

PutTileNo:	       ; Puts tile A at B,C (B,C = screen coordinates)
 ld h,0
 ld l,a
 add hl,hl
 add hl,hl
 add hl,hl	       ; Multiply tile with 8 since each tile is 8 bytes
 ld de,Sprites
 add hl,de	       ; Add with pointer to first tile
 call PutSprite        ; And put the sprite
 ret

CoordConv:	       ; Converts tile coordinates B,C to screen coord B,C
 push de
 push hl
 sla c
 sla c
 sla c		       ; Multiply the Y coordinate with 8
 ld h,0
 ld l,b
 add hl,hl
 add hl,hl
 add hl,hl	       ; Multiply the X coordinate with 8
 ld de,(XScr)
 sbc hl,de	       ; And decrease it with XScr to get the screen coord
 ld b,l
 pop hl
 pop de
 ret

GetBlock:	       ; Gets the block at (HL,A) -> A. HL = addr, B,C = coord
 srl h
 rr l
 srl h
 rr l
 srl h
 rr l		       ; Divide X with 8
 ld b,l
 add hl,hl
 add hl,hl
 add hl,hl	       ; HL = x with the last three bits cleared
 ld de,LevelPtr
 add hl,de	       ; Add HL with the pointer to the level data
 cp 100 	       ; Check if Y is >0  (signed)
 jr c,OnScreen
 ld c,0 	       ; If Y<0, act as Y was 0
 ld a,(hl)	       ; A = top tile of current column
 or a
 ret z
 cp 32
 ld a,0 	       ; If a background tile, act as space (not xor a!)
 ret nc
 inc a		       ; Else set to steel wall (so you can't pass it)
 ret
OnScreen:
 srl a
 srl a
 srl a		       ; Divide Y with 8
 ld c,a
 ld d,0
 ld e,a
 add hl,de	       ; HL = x*8+y
 ld a,(hl)	       ; A = tile at (HL)
 ret

CheckTile:	       ; Returns with Carry if A=foreground tile, else NC
 or a
 ret z
 cp 32
 ret

CheckLeft:	       ; Returns with carry if not possible to move left
 ld hl,(x)
 dec hl
CheckHorz:
 ld a,(y)	       ; HL,A = x,y to the left of Sqrxz
 push af
 call GetBlock	       ; Find out what tile is there
 ld b,a 	       ; B will be the leftupper tile of Sqrxz
 ld c,a 	       ; C will be the leftlower tile of Sqrxz
 pop af
 bit 7,a	       ; Check if Y is negative
 jr nz,CH_Even	       ;  If so, leftlower=leftupper
 and 7		       ; Else check if Y is between two tiles
 jr z,CH_Even	       ;  If not, leftlower=leftupper
 inc hl 	       ; Else leftlower = address after leftupper
 ld c,(hl)	       ; So now C=leftlower tile
CH_Even:
 ld a,b
 call CheckTile        ; Check if foreground tile
 ret c		       ;  If so, return with carry
 ld a,c
 call CheckTile        ; Check if foreground tile
 ret		       ; Return with carry or not carry

CheckRight:
 ld hl,(x)
 ld de,8
 add hl,de	       ; HL = x to the right of Sqrxz
 jr CheckHorz	       ; Continue on the routine above

CheckUp:	       ; Returns with Carry if not possible to move up
 ld a,(y)
 dec a		       ; A = possible new y coordinate
 jr CheckVert	       ; Continue below

CheckDown:	       ; Returns with Carry if not possible to move up
 ld a,(y)
 add a,8	       ; A = possible new y coordinate
 ld bc,0	       ; If the routine ends at the return below, BC must be 0
 cp 64		       ; If the new Y coordinate >=64
 ret nc 	       ;  Then return with no carry, since you can fall forever
CheckVert:
 ld hl,(x)	       ; HL,A = x,y below of Sqrxz
 call GetBlock	       ; Find out what tile is there
 ld b,a 	       ; B will be the leftlower tile of Sqrxz
 ld c,a 	       ; C will be the rightlower tile of Sqrxz
 ld a,(x)
 and 7		       ; Check if X is between two tiles
 jr z,CV_Even	       ;  If not, leftlower = rightlower
 ld de,8	       ; Else add the address with 8 to get the address
 add hl,de	       ; to the right
 ld c,(hl)	       ; And load the tile in that address into C
CV_Even:
 ld a,b
 call CheckTile        ; Check if foreground tile
 ret c		       ;  If so, return with carry
 ld a,c
 call CheckTile        ; Check if foreground tile
 ret		       ; Return with carry or not carry

PutSqrxz:
 ld hl,(x)
 ld de,(XScr)
 or a
 sbc hl,de	       ; Now HL = x screen coordinate of Sqrxz
 ld b,l 	       ; B = x pos on screen
 ld a,(y)
 ld c,a 	       ; C = y pos on screen
 ld (SqrxzBG),bc       ; Store those coordinates in the Background buffer
 ld hl,Sqrxz	       ; HL -> Sqrxz sprites
 ld a,(sqrxzcnt)       ; A = Sqrxz animation counter
 srl a		       ; Divide with two so he doesn't animate too fast
 and 3		       ; Now A = 0-3
 add a,a
 add a,a
 add a,a
 add a,a	       ; Multiply with 16 since each sprite is 16 byte (mask)
 ld d,0
 ld e,a
 add hl,de	       ; Add the sprite offset with the pointer
 ld a,(sqrxzdir)
 or a		       ; Check if Sqrxz is faced to the right
 jr z,FaceRight        ;  If so, no more adding
 ld de,64
 add hl,de	       ; Add with 64 (16*4) to get those sprites
FaceRight:
 ld de,SqrxzBG+2       ; DE -> Sqrxz background storage
 jp PutSprite_MSB      ; Put the sprite (1 byte less than call PS \ ret)

RemoveSqrxz:
 push af
 push bc	       ; These register needs to be saved
 ld bc,(SqrxzBG)       ; B,C = old Sqrxz screen relative coordinates
 ld hl,SqrxzBG+2       ; HL -> Sqrxz background
 call PutSprite        ; Put the background
 pop bc
 pop af
 ret

PutEnemies:
 ld hl,0
 ld (LastSprite),hl    ; Clear the LastSprite pointer (will be updated)
 ld ix,enemyTable      ; IX -> enemy table
 ld de,BGETable        ; DE -> table storing enemy backgrounds
 ld b,8 	       ; Check all 8 positions in the enemy table
RepPutEnemy:
 push bc
 ld a,(ix)
 or a		       ; Check if this position contains any enemy
 jp z,PNext	       ;  If not, try next
 ld (LastSprite),de    ; LastSprite -> Last enemy background information
 push de
 ld h,(ix+3)
 ld l,(ix+2)	       ; HL = enemy x position
 ld de,(XScr)
 sbc hl,de	       ; HL = enemy x position on screen
 push hl
 dec a		       ; A = enemy number (0-4)
 add a,a
 ld hl,EnemyAddr       ; HL -> table of enemy gfx pointers
 ld d,0
 ld e,a
 add hl,de	       ; HL -> pointer to enemy gfx
 ld b,(hl)
 inc hl
 ld h,(hl)
 ld l,b 	       ; HL -> start of enemy graphics for current enemy
 cp 8		       ; Check if fish man
 jr z,HH	       ;  Fishes gfx are stored the same way as hedgehog
 bit 1,a	       ; Check if Blob or Bat
 jr z,EAnimAdd	       ;  Those are stored the same way
 cp 2		       ; Check if hedgehog
 jr z,HH
 bit 0,(ix+1)	       ; Check direction for green man
 jr z,EAnimAdd
 ld de,64	       ; If faced left, add 64 (16x4) to sprite address
 add hl,de
 jr EAnimAdd
HH:		       ; If Fish or Hedgehog, this routine will be reached
 bit 0,(ix+1)
 jr z,NoHHDirAdd
 ld de,32	       ; If faced left, add 32 (16x2) to sprite address
 add hl,de
NoHHDirAdd:
 ld a,(counter)
 bit 4,a
 jr z,AddrFound
 ld de,16	       ; Depending on counter, use the second sprite
 add hl,de
 jr AddrFound
EAnimAdd:	       ; Reached here if Blob, Bat or Green Man
 ld a,(counter)        ; All those have four sprites/direction
 and $18
 add a,a	       ; A will now be 0, 16, 32 or 64
 ld d,0
 ld e,a
 add hl,de
AddrFound:	       ; When reached here, HL -> sprite to put
 pop de 	       ; if enemy is moving
 ld b,e 	       ; B = screen X coordinate to put enemy
 bit 7,d	       ; Check if X coordinate is negative
 jr nz,XOk
 ld a,d
 or a		       ; Check if X>=256
 jr nz,XNotOk	       ;  If so, X not OK
 bit 7,e	       ; Check if X>=128
 jr z,XOk	       ;  If not, X is in range
XNotOk:
 ld b,130	       ; If X>=128, set X to 130 (else wrap could occur)
XOk:
 ld c,(ix+4)	       ; C = sprite Y coordinate
 pop de
 ex de,hl	       ; Now HL -> enemy background storage, DE -> sprite
 ld (hl),b	       ; Store the enemy coordinates
 inc hl
 ld (hl),c
 inc hl
 ex de,hl
 ld a,(ix+6)
 or a		       ; Check if any special stuff is happening
 jr z,PutEnemy	       ;  If not, put the sprite
Special:
 push bc
 push de
 inc a
 ld (ix+6),a	       ; Increase the spec counter
 ld c,(ix)
 dec c		       ; Check if blob
 jr z,KillBlob	       ;  If so, the blob died this frame
 dec c
 dec c		       ; Check if bat
 jr z,KillBat	       ;  Update bat animation
 ld hl,GreenManHide    ; Else it was the green man
 cp 4
 jr c,SpecialPop       ; If spec<4, show half hiding green man
 ld de,16
 add hl,de	       ; HL -> hiding green man
 cp 100
 jr c,SpecialPop       ; If spec<100, show hiding green man
 sbc hl,de	       ; HL -> half hiding green man
 cp 120
 jr c,SpecialPop       ; If spec<120, show half hiding green man
 ld (ix+6),0	       ; Else reset the spec counter - green man is in action
 ld a,(ix+1)
 xor 1		       ; Change the green mans direction
 ld (ix+1),a
 jr SpecialPop
KillBat:
 ld hl,BatSplash       ; HL -> bat splash graphics
 add a,a
 add a,a
 and $F0
 ld d,0
 ld e,a
 add hl,de	       ; HL -> splash gfx this frame
 cp 48
 jr c,SpecialPop       ; If <48, show it
 ld (ix),0	       ; Else the splash animation is over - remove enemy
 ld de,16
 sbc hl,de	       ; HL -> last splash gfx
 jr SpecialPop
KillBlob:
 ld hl,BlobDead        ; HL -> dead blob gfx
 ld (ix),0	       ; Remove enemy from enemy table
 pop de
 pop bc
 call PutSprite_MSB    ; Show dead blob
 jr PutEnemy	       ; And show it again (else the dead blob would disappear)
SpecialPop:
 pop de
 pop bc
PutEnemy:
 call PutSprite_MSB    ; Show sprite
 ex de,hl
 ld de,8
 add hl,de
 ex de,hl	       ; HL -> next position in sprite background table
PNext:
 ld bc,7
 add ix,bc	       ; IX -> next position in enemy table
 pop bc
 dec b
 jp nz,RepPutEnemy     ; Repeat until all enemies have been put to screen
 ret

RemoveEnemies:
 push af
 ld hl,flags
 bit 5,(hl)	       ; Check if this routine has already been called
 jr nz,RemDone
 set 5,(hl)	       ; Set that flag now else
 ld hl,(LastSprite)    ; HL -> last enemy that has been put to screen
 ld a,h
 or l		       ; If 0, no enemies was on screen
 jr z,RemDone
RemNextSprite:
 ld b,(hl)
 inc hl
 ld c,(hl)
 inc hl 	       ; Now B,C = previous enemy coordinates, HL->background
 call PutSprite        ; Show the background
 ld de,BGETable+2
 call _cphlde	       ; Check if all backgrounds have been put to screen
 jr z,RemDone
 ld de,-12
 add hl,de	       ; HL -> next background to be put to screen
 jr RemNextSprite
RemDone:
 pop af
 ret

ScrollLeft:
 ld b,a
 ld a,(XBlock)
 add a,15
 ld hl,LevelSize
 cp (hl)	       ; Check if end of level reached
 ld a,b
 ret nc 	       ; If so, return without scrolling
 di		       ; Disable interrupts so IY can be used
 ld hl,SqrxzBG+1
 dec (hl)	       ; Update Sqrxz screen X coordinate
 push af
 push iy
Modify_2:
 ld hl,0	       ; HL -> end of video_mem+1
 ld c,64	       ; 64 rows to scroll
RepScrollR1:
 dec hl
 ld b,3 	       ; 3x5 bytes to scroll each row
RepScrollR2:	       ; The actual scroll loop
 dec hl
 rl (hl)	       ; It's faster to use 3x5 rotates instead of
 dec hl 	       ; 1x15 since the djnz is called much less frequently
 rl (hl)
 dec hl 	       ; Every bit of speed is needed!
 rl (hl)
 dec hl
 rl (hl)
 dec hl
 rl (hl)
 djnz RepScrollR2      ; Repeat the first scroll loop
 dec c
 jr nz,RepScrollR1     ; And scroll the next row
 ld hl,(XScr)
 inc hl
 ld (XScr),hl	       ; Increase the scroll X coordinate
 ld a,l
 ld hl,XBlock
 and $07	       ; Check if the leftmost block has changed
 jr z,Even
 scf		       ; If not, there are 16 tiles left<->right
 .db $3E	       ; Opcode for LD A,n - will skip the next instruction
Even:
 inc (hl)	       ; If so, increase XBlock
 ld a,(hl)	       ; A = x block tile to left
 adc a,14	       ; Add with 14 (or 15) to get the x block tile to right
 ld h,0
 ld l,a
 add hl,hl
 add hl,hl
 add hl,hl
 ld de,LevelPtr
 add hl,de	       ; HL -> level data of the blocks to be put to the right
 push hl
 pop iy
 ld a,(XScr)
 and $07
 ld hl,ExpTable
 ld d,0
 ld e,a
 add hl,de
 ld d,(hl)	       ; D = bit value to mask out of tile sprite data
 ld c,8 	       ; 8 tiles to put
Modify_3:
 ld ix,$FC0E
RepFillRight:
 push de
 ld a,(iy)	       ; A = tile to put
 inc iy
 bit 7,a	       ; Check if an enemy
 jr z,SL_Tile
 push bc	       ; Prepare to add an enemy to the enemy table
 push hl
 push af
 ld a,8
 sub c
 add a,a
 add a,a
 add a,a
 ld c,a 	       ; C = y coordinate of enemy
 ld a,(XBlock)
 add a,15
 ld h,0
 ld l,a
 add hl,hl
 add hl,hl
 add hl,hl	       ; HL = x coordinate of enemy
 pop af
 and $7F
 call NewEnemy	       ; Add enemy to table
 cp 5		       ; Check if it was a fish
 ld a,109	       ; If so, then the tile should be replace with water
 jr z,FillWater
 xor a		       ; Else an empty tile
FillWater:
 ld (iy-1),a	       ; Store the new tile
 pop hl
 pop bc
SL_Tile:
 ld h,0
 ld l,a
 add hl,hl
 add hl,hl
 add hl,hl
 ld de,Sprites
 add hl,de	       ; HL -> tile to put
 pop de
 ld b,8 	       ; 8 rows in the sprite
SR_Fill:
 ld a,(hl)
 inc hl
 and d		       ; This masks out the only bit that will be put
 jr z,RemPixel	       ; If bit cleared, remove the last pixel
 set 0,(ix)	       ; Else set it
 jr SR_NextY
RemPixel:
 res 0,(ix)	       ; Remove pixel
SR_NextY:
 push de
 ld de,16
 add ix,de	       ; IX -> next row in video mem
 pop de
 djnz SR_Fill	       ; Proceed with next row in tile
 dec c
 jr nz,RepFillRight    ; And proceed with next tile
SR_Done:
 pop iy
 pop af
 ei
 ret

NewEnemy:  ; A - type, C - y coord, HL - x coord
 push af
 push bc
 push de
 push hl
 ld hl,enemyTable      ; HL -> enemy table
 ld b,8 	       ; Check 8 positions (max)
 ld d,a
SearchFreeEntry:
 ld a,(hl)
 or a		       ; Check if position busy (type>0)
 jr nz,NextEnt	       ;  Yes it was, try next entry
 ld (hl),d	       ; Store the enemy type
 inc hl
 ld (hl),1	       ; Followed by the direction (always 1=Left)
 pop de 	       ; Pop the x coordinate
 inc hl
 ld (hl),e	       ; Store the LSB of it
 inc hl
 ld (hl),d	       ; Followed by the MSB
 inc hl
 ld (hl),c	       ; Then store the y coordinate
 inc hl
 ld (hl),0	       ; Clear the fall counter
 inc hl
 ld (hl),0	       ; And last the 'spec' byte
 jr NE_Done
NextEnt:
 push de
 ld de,7
 add hl,de	       ; Add with 7 to reach next entry
 pop de
 djnz SearchFreeEntry  ; And try to find a new spot if possible
 pop hl
NE_Done:
 pop de
 pop bc
 pop af
 ret

LoadLevel:
 ld hl,0	       ; Prepare to clear a lot of variables
 ld (XScr),hl
 ld (jump),hl
 ld (dead),hl
 ld (sqrxzcnt),hl
 ld (portc),hl
 ld (lastsprite),hl
 ld (x),a
 xor a
 ld (animptr),a
 ld (animcnt),a
 ld (deadcnt),a
 ld (y),a
 ld (XBlock),a
 ld hl,AnimTable
 ld bc,39
 ld (hl),$FF
 call FillChar	       ; Clear the animation table (with $FF)
 ld hl,enemyTable
 ld bc,55
 call ClearMem	       ; Clear enemy table
 ld hl,_plotSScreen
 ld bc,1023
 call ClearMem	       ; Clear graphmemory (virtual screen)

 call _clrLCD
 ld hl,LevelPtr
 ld a,(LevelSize)
 ld c,a
 ld a,(start_x)
 or a		       ; Check if a lever has been pulled
 jr z,NoLeverPulled
 push af	       ; If so, the left edge of the screen should be
 sub 2		       ; start_x-2
 jr nc,NoProbL
 xor a		       ; But not less than 0
NoProbL:
 ld b,a
 add a,14
 cp c		       ; Check so the right edge won't be beyond size of level
 ld a,b
 jr c,NoProbR
 ld a,c
 sub 15 	       ; Fix if necessary
NoProbR:
 ld (XBlock),a	       ; Store the start screen location in XBlock
 ex de,hl
 ld h,0
 ld l,a
 add hl,hl
 add hl,hl
 add hl,hl
 ld (XScr),hl	       ; and in XScr as well
 ex de,hl
 add hl,de
 pop af
 push hl
 ld h,0
 ld l,a
 add hl,hl
 add hl,hl
 add hl,hl
 ld (x),hl
 pop hl
 ld a,(start_y)
 add a,a
 add a,a
 add a,a
 ld (y),a	       ; Set Sqrxz Y coordinate according to start_y
NoLeverPulled:
 ld bc,0	       ; B,C = current x,y coordinates
RepShowLevel:
 ld a,(hl)
 bit 7,a	       ; Check if tile to put is an enemy
 jr z,PlainTile
 push hl
 and $7F
 jr nz,EnemySt	       ; If it was an enemy, put it out
 ld a,(start_x)
 or a
 jr nz,Restore	       ; If lever pulled, don't update Sqrxz x,y coordinates
 ld a,c
 ld (y),a
 ld h,0
 ld l,b
 ld (x),hl	       ; Else store Sqrxz start coordinates
 jr Restore
EnemySt:
 ld h,0
 ld l,b
 ld de,(XScr)
 add hl,de
 call NewEnemy	       ; Add enemy to enemytable
 cp 5		       ; Check if it was a fish
 jr nz,Restore
 ld a,109	       ; If it was, water should be there as background
 jr FRestore
Restore:
 xor a
FRestore:
 pop hl
 ld (hl),a	       ; Change the tile to space (or water if fish)
PlainTile:
 inc hl
 push hl
 ld h,0
 ld l,a
 add hl,hl
 add hl,hl
 add hl,hl
 ld de,Sprites
 add hl,de	       ; HL -> tile to put
 call PutSprite        ; And put the sprite
 pop hl
 ld a,c
 add a,8	       ; Increase the Y coordinate with 8
 ld c,a
 cp 64
 jr nz,RepShowLevel    ; If not 64, continue showing level
 ld c,0 	       ; Else reset Y coordinate
 ld a,b
 add a,8	       ; And add the X coordinate
 ld b,a
 cp 120
 jr nz,RepShowLevel    ; If less than 120, continue with the level

 ld a,(timeleft+1)     ; A = MSB of time to beat level
 ld c,a
 ld b,60	       ; Size of water bar
Modify_4:
 ld hl,$FFFF
 ld de,-16
ShowTime:
 ld a,c
 or a		       ; Check if top of time bar reached
 ld a,%00010000        ; If so, use this byte
 jr z,NoTimeBar
 dec c
 ld a,%00010110        ; Else use this byte to show bar
NoTimeBar:
 ld (hl),a
 add hl,de
 djnz ShowTime	       ; Repeat until the whole bars have been shown
 ld hl,479
 ld (waterbar),hl      ; Set water bar max value
 ret

ChooseWorld:	       ; This routine lets the user choose world
 call _RAM_PAGE_7
 call _clrLCD
 ld (_curRow),de       ; DE gets cleared by _clrLCD
 ld hl,WorldChoice
 set textInverse,(iy+textflags)
 call _puts
 call ShowCInfo
 res textInverse,(iy+textflags)
 ld hl,noWorlds
 ld bc,167
 call ClearMem	       ; Clear list of worlds found
 ld ix,worlds
 ld hl,$BFFF	       ; Prepare scanning variables
SearchWorlds:
 ld de,(_endSymbTable)
 inc de
 or a
 sbc hl,de
 jr c,SearchDone
 add hl,de
 push hl
 ld a,(hl)
 cp $0C
 jr z,StringFound
ContinueSearch:
 pop hl
 dec hl
 dec hl
 dec hl
 dec hl
 dec hl
 ld b,(hl)
 inc b
SkipName:
 dec hl
 djnz SkipName
 jr SearchWorlds
StringFound:
 dec hl
 ld e,(hl)
 dec hl
 ld d,(hl)
 dec hl
 ld a,(hl)
 ex de,hl
 call _inc_ptr_ahl
 call _inc_ptr_ahl
 ld c,a
 push hl
 call _GETB_AHL
 cp 2		       ; Check first header byte
 pop hl
 ld a,c
 jr nz,NotSqrxzWorld   ; If not 2 (indicating Sqrxz level), continue search
 call _inc_ptr_ahl
 push hl
 call _GETB_AHL
 pop hl
 cp 2		       ; Check second header byte
 jr nc,NotSqrxzWorld   ; If not 0 or 1, no Sqrxz level - continue search
 ld (ix),h	       ; Else store the address to that level
 ld (ix+1),l
 ld (ix+2),c
 ld de,(lastW_Lo)
 call _cphlde
 jr nz,NotLastWorld
 ld a,(lastW_Hi)
 cp c
 jr nz,NotLastWorld
 ld a,(noWorlds)
 push af
 and 7
 ld (curItem),a
 pop af
 and $F8
 ld (top),a
NotLastWorld:
 inc ix
 inc ix
 inc ix
 ld hl,noWorlds
 inc (hl)	       ; Increase number of worlds found
NotSqrxzWorld:
 call _RAM_PAGE_7
 jr ContinueSearch     ; Continue search
SearchDone:
 ld a,(noWorlds)
 or a		       ; Check if no worlds found
 jr nz,ShowWorlds
 ld hl,$0304
 ld (_curRow),hl
 ld hl,NoWorldsTxt
 call _puts
 call WaitEnter
 jp Abort
ShowWorlds:
 ld hl,$FC80
 ld bc,783
 call ClearMem	       ; Clear part of video memory
 ld hl,$0803
 ld (_penCol),hl
 ld hl,worlds
 ld a,(top)	       ; Top = top of list
 ld d,a
 add a,a
 add a,d
 ld d,0
 ld e,a
 add hl,de	       ; HL -> pointers to worlds
 ld b,8 	       ; Max 8 worlds to be shown at the same time
ShowWorld:
 ld d,(hl)
 inc hl
 ld e,(hl)	       ; DE = pointer to world data
 inc hl
 ld a,(hl)	       ; A = RAM block
 inc hl
 or a
 jr z,EndOfList        ; If A=0, end of list reached
 push bc
 push hl
 ex de,hl
 ld de,_OP1
 ld b,33
 call ReadVar
 ex de,hl
 inc hl 	       ; Skip level type
 inc hl 	       ; Skip first byte of no levels (always 0)
 ld a,(hl)	       ; A = number of levels in this world
 inc hl
 push af
 call _vputs	       ; Show world name
 ld a,$68
 ld (_penCol),a        ; Update X coordinate
 pop af
 call MDispA	       ; And display number of levels
 ld hl,_penCol
 ld (hl),$03	       ; Set X coordinate to 3
 inc hl
 ld a,(hl)
 add a,6	       ; And increase the Y coordinate with 6
 ld (hl),a
 pop hl
 pop bc
 djnz ShowWorld        ; Repeat showing the list
EndOfList:
 ld a,8
 sub b		       ; A = number of worlds in current list
 ld (noItems),a
 ld a,(curItem)
UpdateChoice:
 ld (curItem),a
 call MarkItem	       ; Mark the current selection
Choose:
 call _getky
 cp _kyExit
 jr z,Abort
 cp _kyMore
 jr z,NextPage
 cp _kyEnter
 jp z,Select
 cp _kyAlpha
 jp z,PracticeWorld
 dec a
 jr z,CursorDown
 cp 3
 jr nz,Choose
 ld a,(curItem)
 call MarkItem	       ; Unmark the current selected world
 or a		       ; Check if at top of list
 jr nz,NoWarp	       ; If not, decrease
 ld a,(noItems)        ; Else A=bottom of list+1
NoWarp:
 dec a
 jr UpdateChoice
CursorDown:
 ld a,(curItem)
 call MarkItem	       ; Unmark the current selected world
 inc a
 ld hl,noItems
 cp (hl)	       ; Check if bottom of list passed
 jr nz,UpdateChoice
 xor a		       ; If so, current world is 0
 jr UpdateChoice
Abort:
 pop hl
QuickQuit:
 im 1		       ; Remove the installed interrupt

 xor a
 ld hl,mode
 call _SET_ABS_SRC_ADDR
 ld hl,5
 call _SET_MM_NUM_BYTES
 ld hl,_asapvar
 rst 20h
 rst 10h
 ex de,hl
 ld a,b
 ld de,mode-_asm_exec_ram+4
 add hl,de
 adc a,0
 call _SET_ABS_DEST_ADDR
 call _mm_ldir

; ld hl,ProgName	; All this stuff is to make sure the options are saved
; rst 20h		; and resume stuff are stored in the variable
; rst 10h
; ex de,hl		; HL -> start of variable
; ld a,b
; ld de,mode-$D748+4	; DE = offset to data (4 byte prgm header)
; add hl,de
; adc a,0		; Next block if necessary
; ld de,mode		; DE = start of bytes to copy
; ld b,2		; 2 bytes to copy
;RepCopyPrgm:
; push af
; push hl
; call $46C3
; ld a,(de)		; Read
; ld (hl),a		; And save it in the real variable
; pop hl
; pop af
; call $4637
; inc de
; djnz RepCopyPrgm

 res textInverse,(iy+textflags)
 set appTextSave,(iy+appflags)


 ld hl,_plotSScreen
 ld bc,1023
 call ClearMem

 call _clrScrn
 call _homeup
 ret		       ; Return to TI-OS / Shell

NextPage:
 ld a,(top)
 add a,8	       ; Add top of list with 8
 ld hl,noWorlds
 cp (hl)	       ; Check if not that many worlds
 jr c,ShowW
 xor a		       ; If so, start from page 0
ShowW:
 ld (top),a
 xor a
 ld (curItem),a
 jp ShowWorlds

PracticeWorld:
 ld a,1
 ld (practice),a
Select:
 ld a,(top)
 ld hl,curItem
 add a,(hl)
 ld hl,worlds
 ld d,a
 add a,a
 add a,d
 ld d,0
 ld e,a
 add hl,de	       ; HL -> pointer to current world
 ld d,(hl)
 inc hl
 ld e,(hl)
 inc hl
 ld a,(hl)	       ; A, DE = world to play
 ex de,hl
 ld (lastW_Lo),hl
 ld (lastW_Hi),a
 ld b,5
 ld de,tmpWorld
CopyWorld:
 push bc
 push af
 push hl
 push de
 call _GETB_AHL
 ld de,_plotSScreen
 ld bc,1024
RepCopy:
 ld a,(hl)
 ld (de),a
 inc de
 inc hl
 ld a,h
 cp $C0
 jr nz,SamePage
 in a,(6)
 inc a
 or $40
 out (6),a
 ld hl,$8000
SamePage:
 dec bc
 ld a,b
 or c
 jr nz,RepCopy
 ld a,$41
 out (6),a
 pop de
 ld hl,_plotSScreen
 ld bc,1024
 ldir
 pop hl
 pop af
 ld bc,1024
 add hl,bc
 adc a,0
 pop bc
 djnz CopyWorld
 ret

MarkItem:
 push af
 ld h,96
 ld l,a
 call _mulhl
 ld de,$FC90
 add hl,de	       ; HL -> start of memory to invert
 ld b,96
InvertRow:
 ld a,(hl)
 cpl		       ; Invert byte
 ld (hl),a
 inc hl
 djnz InvertRow
 pop af
 ret

ChangeOptions:	       ; Lets the user change the game options
 call ShowTitle
 set 3,(iy+$05)
 call ShowCInfo
 res 3,(iy+$05)
 ld hl,$1D10
 ld (_penCol),hl
 ld hl,OptionsTxt
 call _vputs
 ld de,$2510
 ld (_penCol),de
 call _vputs
 ld de,$1510
 ld (_penCol),de
 ld a,(gaming)
 or a
 jr z,NoPause
 ld hl,ResumeTxt
NoPause:
 call _vputs
 call UpdateSpeedOpt
 call UpdateModeOpt
 set 3,(iy+$05)
ResetOpt:
 xor a
 ld (curOpt),a
UpdOpt:
 call UpdateCurOpt
OptLoop:
 call _getky
 cp _kyExit
 jp z,Abort
 cp _kyEnter
 jr z,EntPressed
 ld hl,curOpt
 dec a
 jr z,OptDown
 cp 3
 jr nz,OptLoop
 call UpdateCurOpt
 ld a,(hl)
 dec (hl)
 or a
 jr nz,UpdOpt
 ld (hl),2
 jr UpdOpt
OptDown:
 call UpdateCurOpt
 ld a,(hl)
 inc (hl)
 cp 2
 jr nz,UpdOpt
 jr ResetOpt
EntPressed:
 ld a,(curOpt)
 or a
 jr z,OptDone
 dec a
 jr z,ChangeMode
 ld hl,$2556
 call ClearOpt
 ld hl,speed
 ld a,(hl)
 dec (hl)
 or a
 jr nz,UpdSpe
 ld (hl),2
UpdSpe:
 call UpdateSpeedOpt
 jr OptLoop
ChangeMode:
 ld hl,$1D56
 call ClearOpt
 ld hl,mode
 ld a,(hl)
 xor 1
 ld (hl),a
 call UpdateModeOpt
 jr OptLoop
OptDone:
 res 3,(iy+$05)
 ld a,(speed)
 add a,2
 ld (speedC+1),a       ; Selfmodifying code - updates a CP in the source
 ld hl,ModifyAddr      ; HL -> list of pointers in the source to modify
 ld de,ModifyVal       ; DE -> list of addresses to be updated
 ld b,8 	       ; The lists contains 8 word
ModifyCode:	       ; Prepare to update a lot of pointers
 push bc
 push de
 push hl
 call _ldhlind
 ex de,hl
 call _ldhlind
 ld a,(mode)
 or a		       ; Check if virtual screen was selected
 jr z,DirectVidMem
 ld bc,$CDFA	       ; If so add with GRAPHMEM-VIDEOMEM
 add hl,bc
DirectVidMem:
 ex de,hl
 ld (hl),e	       ; Store the word in the source
 inc hl
 ld (hl),d
 pop hl
 pop de
 pop bc
 inc de
 inc de
 inc hl
 inc hl
 djnz ModifyCode
 ld a,(mode)
 ld hl,Modify_B+1      ; HL -> a JR instruction to be changed
 ld (hl),11	       ; If no virtual screen, JR 11 will skip page flipping
 or a
 ret z
 ld (hl),0	       ; Else a JR 0 will enable the skipping
 ret

ClearOpt:
 ld (_penCol),hl
 ld hl,_OP1
 ld (hl),32
 ld bc,10
 call FillChar
 ld (hl),0
 ld b,3
RepClear:
 ld hl,_OP1
 call _vputs
 djnz RepClear
 ret

UpdateSpeedOpt:        ; Show current screen option
 ld hl,$2556
 ld (_penCol),hl
 ld hl,SpeedOpt
 ld a,(speed)
 add a,a
 add a,a
 add a,a
 ld d,0
 ld e,a
 add hl,de
 call _vputs
 ret

UpdateModeOpt:	       ; Show current mode selection
 ld hl,$1D56
 ld (_penCol),hl
 ld hl,ModeOpt
 ld a,(mode)
 add a,a
 add a,a
 ld d,0
 ld e,a
 add hl,de
 call _vputs
 ret

UpdateCurOpt:
 push hl
 ld hl,$FD51
 ld a,(curOpt)
 ld d,a
 ld e,0
 srl d
 rr e
 add hl,de
 ld c,7
InvRow:
 ld b,14
InvByte:
 ld a,(hl)
 cpl
 ld (hl),a
 inc hl
 djnz InvByte
 inc hl
 inc hl
 dec c
 jr nz,InvRow
 pop hl
 ret

ShowTitle:
 call _clrLCD
 ld hl,$FC00
 ld bc,111
 ld (hl),$FF
 call FillChar
 ld hl,$0001
 ld (_penCol),hl
 ld hl,Title
 set 3,(iy+$05)
 call _vputs
 res 3,(iy+$05)
 ret

ShowCInfo:
 ld hl,$FF90
 ld bc,112
 ld (hl),$FF
 call FillChar
 ld hl,$3906
 ld (_penCol),hl
 ld hl,CInfo
 call _vputs
 ret

ClearMem:
 ld (hl),0	       ; Clears BC+1 chars from HL
FillChar:	       ; Fills BC+1 chars from HL with (HL)
 ld d,h
 ld e,l
 inc de
 ldir
 ret

ReadVar:	       ; Reads B bytes from RAM to DE (HL=address, A=block)
 push de
ContRead:
 push af
 push hl
 call _GETB_AHL
 ld (de),a
 inc de
 pop hl
 pop af
 call _inc_ptr_ahl
 djnz ContRead
 pop de
 ret

WaitEnter:
 push bc
 push de
 push hl
WEnter:
 call _getkey
 cp kEnter
 jr nz,WEnter
 pop hl
 pop de
 pop bc
 ret

ExtractA:
 ld h,0
 ld l,a
 ld de,_OP1+3
 xor a
 ld (de),a
 ld b,3
UnpackNo:
 call _hldiv10
 add a,48
 dec de
 ld (de),a
 djnz UnpackNo
 ret

DispHL:
 push af
 push bc
 push de
 push hl
 ld b,5
 ld de,_OP1+5
 xor a
 ld (de),a
RepDispHL:
 call _hldiv10
 dec de
 add a,48
 ld (de),a
 djnz RepDispHL
 ex de,hl
 call _puts
 pop hl
 pop de
 pop bc
 pop af
 ret

MDispA:
 call ExtractA
 ld b,2
SkipZeros:
 ld a,(de)
 cp 48
 jr nz,SZ_Done
 ld hl,_penCol
 ld a,(hl)
 add a,4
 ld (hl),a
 inc de
 djnz SkipZeros
SZ_Done:
 ex de,hl
 jp _vputs

DispA:
 call ExtractA
 push de
 ld b,2
RemoveZeros:
 ld a,(de)
 cp 48
 jr nz,RZ_Done
 ld a,32
 ld (de),a
 inc de
 djnz RemoveZeros
RZ_Done:
 pop hl
 jp _puts

IntCounter:	       ; Interrupt handler which increases a timer
 ex af,af'
 exx
 ld hl,timer
 inc (hl)
 in a,(3)
 rra
 push af
IntGetKey:	       ; For some reason, call $4064 doesn't work
 call nc,$01A1	       ; Using selfmodifying code to patch the address
 pop af
 ld a,9
 adc a,0
 out (3),a
 ld a,$0B
 out (3),a
 exx
 ex af,af'
 ei
 reti
IntEnd:

#include ascr.h

mode:
 .db 0	   ; 0=Direct videomem, 1=Virtual screen

speed:
 .db 1	   ; 0=fast, 1=normal, 2=slow

LastW_Lo:
 .dw 0
LastW_Hi:
 .db 0

ExpTable:
 .db $01,$80,$40,$20,$10,$08,$04,$02,$01

ArcData:
 .db 0,0,1,0,0,1,0,1,1,1,2,1,1,2,1,2,2,2,3,3,3,2,3,4,4,4,4

EArc:
 .db 0,1,0,1,2,3,2,3,4,5,6,5,8,8

ModifyAddr:
 .dw Modify_1+1,Modify_2+1,Modify_3+2
 .dw Modify_4+1,Modify_5+1,Modify_6+1
 .dw Modify_7+1,Modify_8+1

ModifyVal:
 .dw $FFFF,$0000,$FC0E,$FFFF,$FC00,$FC00,$FFFF,$FFFF

FreeMemText:
 .db "You must have about",0
 .db "2k free memory!",0

WorldChoice:
 .db "SQRXZ   choose world ",0

NoWorldsTxt:
 .db "No worlds found",0

CInfo:
 .db 6,"  ",7,"  move   "
 .db "ENTER  select   "
 .db "EXIT  quits",0

WorldAuthor:
 .db "made by ",0

EnemyAddr:
 .dw Blob,Hedgehog,Bat,Greenman,Fish

LevelTxt:
 .db "Level ",0

OutOfTimeTxt:
 .db " OUT OF TIME ",0

DrownedTxt:
 .db " YOU DROWNED ",0

GameFinTxt:
 .db "CONGRATULATIONS",0
 .db "You have beaten all",0
 .db "levels in this world!",0

OptionsTxt:
 .db "Use virtual screen",0
 .db "Speed",0
 .db "Start game",0
ResumeTxt:
 .db "Resume game",0

ModeOpt:
 .db "NO ",0
 .db "YES",0

SpeedOpt:
 .db "FAST   ",0
 .db "NORMAL ",0
 .db "SLOW   ",0

Title:
 .db "SQRXZ 1.0   - "
Coder:
 .db "by Jimmy Mardell",0
 .db "Title picture by "
 .db "Nicholas Reed",0

PracticeTxt:
 .db "Practice Mode",0
 .db 6,"  ",7,"  change level   "
 .db "ENTER  select",0

TitlePic:
#include sqtitle.huf

#include sqrxzgfx.h
#include huffextr.h

 .end
