;##################################################################
;
;   Phoenix-Z80 (Collisions between enemies and player's bullets)
;
;   Programmed by Patrick Davidson (pad@calc.org)
;        
;   This program is in the public domain.  There is no warranty.
;
;   This file was last modified September 2, 2000.  
;
;##################################################################

;############## Builds table of enemies in GRAPH_MEM
;
; This puts enemies into columns, 0 to 7, on the high bits of their X
; coordinates.  Column 0 is for X coordinates 0 to 15, 1 for 16 to 31, and
; so on.  Then, they are split into groups; for enemies in only 1 column,
; the group is column * 2 + 1, for enemies that spill into the next column,
; the group is column * 2 + 2.
;
; Each group data structure has a one-byte size, followed by a list of
; pointers to each enemy in the group.  The first column starts at
; GRAPH_MEM+48, the next at GRAPH_MEM+96, and so on.  Empty columns are put
; at the edges to allow proper checking of side bullets.  This allows up to
; 23 entries per column, more than enough to accomodate all enemies.

hit_enemies:
        xor     a                       ; Initialize all columns as empty
        ld      b,18
        ld      de,48
        ld      hl,GRAPH_MEM
loop_empty_columns:
        ld      (hl),a
        add     hl,de
        djnz    loop_empty_columns

        ld      hl,&e_array             ; HL -> enemy
        ld      b,e_num                 
build_table_loop:
        ld      a,(hl)                  ; Skip empty enemy entries
        or      a
        jr      z,none_here

        inc     hl
        inc     hl
        inc     hl
        ld      a,(hl)                  ; D = enemy X coordinate
        inc     hl
        ld      e,(hl)                  ; E = enemy Y coordinate
        dec     hl
        dec     hl
        dec     hl
        dec     hl                      ; HL -> width
        or      a
        jr      z,none_here             ; Skip non-ready enemies

        call    &which_group            ; Computer group number in A

        ld      c,a
        add     a,a                     ; A = 2 * group number
        add     a,c                     ; A = 3 * group number
        add     a,a
        add     a,a                     ; A = 12 * group number

        push    hl

        ld      l,a
        ld      h,0                     ; HL = 12 * group number
        add     hl,hl
        add     hl,hl                   ; HL = 48 * group number
        ld      de,GRAPH_MEM
        add     hl,de                   ; HL -> group data structure

        ld      a,(hl)                  ; A = group size before insertion
        inc     (hl)                    ; increment group size
        add     a,a
        inc     a
        call    &ADD_HL_A               ; HL = new entry in group     

        pop     de                      ; DE -> enemy
        ld      (hl),e                  ; store DE in group
        inc     hl
        ld      (hl),d
        ex      de,hl                   ; HL -> enemy

none_here:
        ld      de,e_size
        add     hl,de
        djnz    build_table_loop

;############## Process player bullets
;
; This routine will process every player bullet by checking it for collisions
; with appropriate columns.  If it is all in one column, it must be tested
; against three groups, as shown below
;
; Column 2     |     Column 3      |     Column 4
;              |                   |
;               <-----Bullet------>
;     <---Enemy group 6--->
;               <--Enemy group 7-->
;                         <---Enemy group 8--->
;
; If the bullet straddles two columns, it then must be checked against
; five groups of enemies, as indicated below
;
; Column 2     |    Column 3       |     Column 4     |      Column 5
;              |                   |                  |
;                      <--------Bullet-------->
;    <---Enemy group 6--->
;               <--Enemy group 7-->
;                      <-----Enemy group 8---->
;                                   <-Enemy group 9-->
;                                           <---Enemy group 10--->
;
; Even so, this still saves a lot of time over the default collision testing
; strategy, to test all combinations of objects, which would effectively
; require testing with 15 groups in all cases (though without the sorting
; overhead introduced here).

        ld      hl,&pb_array
        ld      b,pb_num

loop_process_bullets:
        ld      a,(hl)
        or      a          
        jr      z,no_bullet

        inc     hl
        inc     hl
        ld      a,(hl)
        inc     hl
        ld      e,(hl)
        dec     hl
        dec     hl
        dec     hl

        call    &which_group            ; A = group number

        push    hl
        push    bc

        bit     0,a
        jr      nz,check_3_groups 
check_5_groups:
        dec     a
        dec     a                       ; A = first group to test against
        ld      b,5                     ; B = number of groups to test

scan_groups:
        ld      c,a
        add     a,a                     ; A = 2 * group number
        add     a,c                     ; A = 3 * group number
        add     a,a
        add     a,a                     ; A = 12 * group number

        push    hl

        ld      l,a
        ld      h,0                     ; HL = 12 * group number
        add     hl,hl
        add     hl,hl                   ; HL = 48 * group number
        ld      de,GRAPH_MEM
        add     hl,de                   ; HL -> group data structure

        ex      de,hl                   ; DE -> group data structure
        pop     hl

loop_scan_groups:
        push    hl                      ; Save pointer to bullet
        push    de                      ; Save pointer to group data
        push    bc                      ; Save loop counter
        call    &list_collision_check   ; Collision check this group
        pop     bc                      ; Restore loop counter
        pop     de                      ; Restore group data pointer
        ld      hl,48
        add     hl,de
        ex      de,hl                   ; HL -> next group
        pop     hl                      ; Restore bullet pointer
        djnz    loop_scan_groups

        pop     bc
        pop     hl

no_bullet:
        ld      de,pb_size
        add     hl,de
        djnz    loop_process_bullets
        ret

check_3_groups:
        dec     a
        ld      b,3
        jr      scan_groups

;############## Determine which group object is in
;
; Determines which group an object is in.  A holds the X coordinate, E holds
; the Y coordinate.  Returns results in A.  Modifies A, C, and D.

which_group:
        ld      d,a
        rlca
        rlca
        rlca
        rlca
        and     7
        add     a,a
        inc     a
        ld      c,a
        ld      a,d
        and     15
        add     a,e
        cp      16
        ld      a,c
        ret     c 
        inc     a
        ret

;############## Check bullet for collisions within one group
;
; Checks the bullet at (HL) for collisions with group (DE)

list_collision_check:
        inc     hl
        push    hl
        inc     hl
        push    de
        ld      de,test_coords      ; Copy bullet's coords into test bed
        ldi
        ldi
        ldi
        ldi

        pop     hl
        ld      a,(hl)
        or      a
        jr      z,exit_cc_list
        inc     hl                  ; HL -> group data
        ld      b,a                 ; B = number in group

loop_column_scan:
        push    hl
        ld      a,(hl)
        inc     hl
        ld      h,(hl)
        ld      l,a                 ; HL -> one enemy in table
        push    hl
        ld      de,e_x
        add     hl,de
        call    &collision_check
        pop     hl
        jr      c,collision
        pop     hl
        inc     hl
        inc     hl
        djnz    loop_column_scan

exit_cc_list:
        pop     hl
        ret

collision:
        pop     de                  ; DE = junk
        pop     de                  ; DE -> bullet + 1
        ex      de,hl               ; DE -> enemy, HL -> bullet 

        ld      c,(hl)              ; C = bullet damage
        dec     hl
        ld      (hl),0              ; Delete bullet

        ld      hl,e_pwr
        add     hl,de               ; HL -> enemy damage
        ld      a,(hl)
        sub     c
        ld      (hl),a
        jr      nc,collision_done   ; If enemy not destroyed

        ld      a,-5                ; New enemy type (exploding)
        ld      (de),a

        ld      hl,e_img
        add     hl,de               ; HL -> enemy image
        ld      de,&image_explosion
        ld      (hl),e
        inc     hl
        ld      (hl),d

collision_done:
        pop     af
        pop     af
        pop     af
        pop     af
        pop     bc
        pop     hl
        jr      no_bullet
