// Maze3D -- Z80 RAYCASTING ENGINE
// Mapping and Trig tables builder
// Copyright(C)1999 Clem Vasseur
// Email:darkclem@wanadoo.fr

#define	TEXTURE_HEIGHT	16
#define	SCREEN_HEIGHT	64
#define ANGLE_RATIO	1.40625
#define	MAX_SCALE	128

#include <stdio.h>
#include <math.h>

FILE *out;
unsigned char scale, name_str[12];

void printHeader(void)
{
  fprintf(out, ";=================================================\n");
  fprintf(out, "; MAZE-3D RAYCASTING ENGINE Copyright(C)1998-1999\n");
  fprintf(out, ";-------------------------------------------------\n");
  fprintf(out, "; %s: part of Maze-3D source code\n", name_str);
  fprintf(out, "; Maze-3D Raycasting Engine version 0.5 beta\n");
  fprintf(out, "; Written by Clem Vasseur <darkclem@wanadoo.fr>\n");
  fprintf(out, "; Icarus Productions <http://icarus.ticalc.org>\n");
  fprintf(out, ";=================================================\n");
}

void printM(void)
{
  if(scale < 10)
    fprintf(out, "M00%d", scale);
  else
    if(scale < 100)
      fprintf(out, "M0%d", scale);
    else
      fprintf(out, "M%d", scale);
}

double round(double arg)
{
  if((arg - floor(arg)) < 0.5)
    return(floor(arg));
  else
    return(ceil(arg));
}

int main(void)
{
  unsigned char i, n, array[MAX_SCALE];
  unsigned short size, k;
  double step, temp, j, ang;

  printf("\n +---------------------------------+");
  printf("\n | Maze3D -- Z80 RAYCASTING ENGINE |");
  printf("\n | Mapping and Trig tables builder |");
  printf("\n |  Copyright(C)1999 Clem Vasseur  |");
  printf("\n |    Email:darkclem@wanadoo.fr    |");
  printf("\n +---------------------------------+\n\n");

/***************************
    CREATE MAPPING.INC
***************************/

  if ((out = fopen("Mapping.inc", "w")) == NULL)
  {
    fprintf(stderr, "Error opening output file: Mapping.inc\n");
    return 1;
  }

  strcpy(name_str, "Mapping.inc");
  printHeader();
  fprintf(out, "\nMappingPtrs:\n");

  for(scale = 2; scale <= MAX_SCALE; scale += 2)
  {
    if(((scale-2) % 16) == 0)
      fprintf(out, "\n	.dw ");
    else
      fprintf(out, ",");
    printM();
  }

  fprintf(out, "\n\n");
  size = 0;

  for(scale = 2; scale <= MAX_SCALE; scale += 2)
  {
    step = TEXTURE_HEIGHT / (double)scale;
    printM();
    fprintf(out, ":	.db ");
    temp = j = 0;

    for(i = 0; i < scale; i++)
    {
      array[i] = round(j) - round(temp);
      temp = j;
      j += step;
    }
    if(scale <= SCREEN_HEIGHT)
    {
      for(i = 0; i < (scale - 1); i++)
        fprintf(out, "%d,", 2 * array[i]);
    }
    else
    {
      n = 0;
      for(i = 0; i < (scale - SCREEN_HEIGHT) / 2; i++)
        n += array[i];
      fprintf(out, "%d,", 2 * n);
      for(n = 0; n < (SCREEN_HEIGHT - 2); n++)
        fprintf(out, "%d,", 2 * array[i++]);
    }
    fprintf(out, "%d\n", 2 * array[i]);

    if(scale < SCREEN_HEIGHT)
      size += scale;
    else
      size += SCREEN_HEIGHT;
  }
  fprintf(out, "\n;Size: %d bytes\n", size + MAX_SCALE);
  fclose(out);

  printf("File Mapping.inc created\n");

/***************************
      CREATE TRIG.INC
***************************/

  if ((out = fopen("Trigo.inc", "w")) == NULL)
  {
    fprintf(stderr, "Error opening output file: Trig.inc\n");
    return 1;
  }

  strcpy(name_str, "Trigo.inc");
  printHeader();

  fprintf(out, "\nSin64Table:\n");

  for(k = 0; k <= 255; k++)
  {
    if((k % 16) == 0)
      fprintf(out, "\n	.db ");
    else
      fprintf(out, ",");
    ang = (double)k * ANGLE_RATIO;
    fprintf(out, "%3.0f", 64 * sin(ang * (PI / 180)));
  }

  fprintf(out, "\n\n\nFishbowlTable:\n");

  for(k = 32; k < 96; k++)
  {
    if((k % 16) == 0)
      fprintf(out, "\n	.db ");
    else
      fprintf(out, ",");
    ang = (double)k * ANGLE_RATIO;
    temp = 256 * sin(ang * (PI / 180));
    if(temp == 256)
      temp = 255;
    fprintf(out, "%3d", (int)temp);
  }

  fprintf(out, "\n\n\nSinInv256Table:\n");

  for(k = 0; k <= 127; k++)
  {
    if((k % 8) == 0)
      fprintf(out, "\n	.dw ");
    else
      fprintf(out, ",");
    ang = (double)k * ANGLE_RATIO;
    temp = 256 / sin(ang * (PI / 180));
    if(ang == 0)
      fprintf(out, "65535");
    else
      fprintf(out, "%5.0f", temp);
  }

  fprintf(out, "\n\n\nTan256Table:\n");

  for(k = 0; k <= 127; k++)
  {
    if((k % 8) == 0)
      fprintf(out, "\n	.dw ");
    else
      fprintf(out, ",");
    ang = (double)k * ANGLE_RATIO;
    temp = 256 * fabs(tan(ang * (PI / 180)));
    if(k == 0)
      fprintf(out, "    0");
    else
      if(k == 64)
        fprintf(out, "65535");
      else
        fprintf(out, "%5.0d", (int)temp);
  }

/*
  fprintf(out, "\n\n\nArcTanTable:\n");

  for(k = 0; k < 128; k+=(1.0 / 128.0))
  {
    if((k % 16) == 0)
      fprintf(out, "\n	.db ");
    else
      fprintf(out, ",");
    ang = (double)k;
    fprintf(out, "%3.0f", (180 / PI) * 15 * atan(ang) * 64.0 / 90.0);
  }
*/

  fprintf(out, "\n\n.end\n");

  fclose(out);
  printf("File Trigo.inc created\n");

  return 0;
}
